#!/usr/bin/env python3
##
## -----------------------------------------------------------------
##    This file is part of WAPT Software Deployment
##    Copyright (C) 2012 - 2024  Tranquil IT https://www.tranquil.it
##    All Rights Reserved.
##
##    WAPT helps systems administrators to efficiently deploy
##    setup, update and configure applications.
## ------------------------------------------------------------------
##

import uuid
import time
import sys
import os
import logging
import subprocess
import glob
import platform
import codecs

if sys.platform == 'win32':
    import win32profile
    import win32api
    import win32ts
    import win32con
    import win32process
    import win32security
    from setuphelpers import HKEY_LOCAL_MACHINE, REG_DWORD, KEY_ALL_ACCESS, reg_openkey_noredir, reg_enum_subkeys, reg_delvalue, reg_setvalue, system32
else:
    from pwd import getpwnam

import re
import wakeonlan

from flask import Blueprint, request, Response
from waptservice.flask_utilities import requires_enterprise

from waptlicences import render_mustache

from waptservice.waptservice_common import (
    wapt_root_dir,
    waptconfig,
    WaptTask,
    WaptAuditPackage,
    WaptUpdateServerStatus,
    register_remote_action,
    render_wapt_template,
    WaptUpgrade,
    check_battery,
    allow_local_token,
)

from waptservice.waptservice_common import _

import setuphelpers
from setuphelpers import makepath

from waptutils import ensure_list, jsondump

logger = logging.getLogger('enterprise')

waptwua_api = Blueprint('waptwua_api', __name__)
waptwua_api.task_manager = None


def get_active_sessions():
    try:
        return [s['SessionId'] for s in win32ts.WTSEnumerateSessions() if s['State'] == 0]
    except:
        session_id = win32ts.WTSGetActiveConsoleSessionId()
        # not logged.
        if session_id == 0xffffffff:
            return []
        else:
            return [session_id]


def start_interactive_process(app_filename, cmdline=None, session_id=None, hide= False, minimize=False):
    """Starts a process in the context of opened interactive session if any"""
    if session_id is None:
        session_id = win32ts.WTSGetActiveConsoleSessionId()
    # not logged.
    if session_id == 0xffffffff:
        return None
    # waptservice is running inside a job, we don't want (can't) this process to be part of the job as it is not the same user.
    CREATE_BREAKAWAY_FROM_JOB =  0x1000000
    priority = win32con.NORMAL_PRIORITY_CLASS | win32con.CREATE_NEW_CONSOLE | CREATE_BREAKAWAY_FROM_JOB
    startup = win32process.STARTUPINFO()
    startup.lpDesktop = r'winsta0\default'
    startup.dwFlags = win32con.STARTF_USESHOWWINDOW
    startup.wShowWindow = win32con.SW_HIDE

    try:
        token = win32ts.WTSQueryUserToken(session_id)
    except:
        token = None

    if minimize:
        cmd = '/c start "wapt" /min "%s" %s' % (app_filename, cmdline)
    elif hide:
        cmd = '/c start "wapt" /B "%s" %s' % (app_filename, cmdline)
    else:
        cmd = '/c start "wapt" "%s" %s' % (app_filename, cmdline)

    if token:
        environment = win32profile.CreateEnvironmentBlock(token, False)
        new_token = win32security.DuplicateTokenEx(token, win32security.SecurityDelegation, win32security.TOKEN_ALL_ACCESS, win32security.TokenPrimary)
        process_info = win32process.CreateProcessAsUser(new_token, os.path.join(win32api.GetSystemDirectory(), 'cmd.exe'), cmd, None, None, True, priority, environment, None, startup)
        if token:
            win32api.CloseHandle(token)
        if new_token:
            win32api.CloseHandle(new_token)
    else:
        process_info = win32process.CreateProcess(os.path.join(win32api.GetSystemDirectory(), 'cmd.exe'), cmd, None, None, True, priority, None, None, startup)
    return process_info

SUMMARY_TEMPLATE = """\
Status:            {{status}}
Installed updates: {{installed}}
Pending updates:   {{pending}}
Discarded updates: {{discarded}}
Reboot required:   {{last_install_reboot_required}}

Last install date:   {{LocalDatetime last_install_date}}
Last install result: {{last_install_result}}

WSUSScan cab date:   {{LocalDatetime wsusscn2cab_date}}
"""

class WaptGPUpdate(WaptTask):
    """Launch gpupdate /force on local host"""

    def __init__(self, **args):
        super(WaptGPUpdate, self).__init__()
        self.notify_server_on_finish = True
        self.target_users = True
        self.target_computer = True

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        self.update_status(_("Start GPUpdate command"))
        self.progress = 0.0
        if sys.platform == 'win32':
            active_sessions = []
            if self.target_users:
                active_sessions = get_active_sessions()
            step_percents = 100.0 / (1.0 + len(active_sessions))
            if self.target_computer:
                self.result = self.run_external('gpupdate /Force /Target:Computer')
                self.progress += step_percents
                self.update_status('GPO updated for Computer')
            if self.target_users:
                for session_id in active_sessions:
                    start_interactive_process(makepath(system32(), 'gpupdate.exe'), '/Force /Target:User', session_id=session_id, hide=True)
                    self.progress += step_percents
                    self.update_status('GPO updated for session %s' % session_id)
        self.update_status(_("Update Site , Dn and groups"))
        self.wapt.get_cache_domain_info(force=True)
        self.progress = 100.0

    def same_action(self, other):
        return False

    def __str__(self):
        return _("Update AD Group Policies status")


class WaptRunSessionSetup(WaptTask):
    """Launch session-setup for all active sessions"""

    def __init__(self, **args):
        super(WaptRunSessionSetup, self).__init__()
        self.priority = 999

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        self.update_status(_("Run Session-setup"))
        self.progress = 0.0
        nb = 0
        if sys.platform == 'win32':
            active_sessions = get_active_sessions()
            for session_id in active_sessions:
                if start_interactive_process(os.path.join(wapt_root_dir, 'waptpythonw.exe'), '-I "%s" session-setup ALL' % os.path.join(wapt_root_dir, 'wapt-get.py'), session_id=session_id, hide=True):
                    nb = nb + 1
        else:
            dict_user = setuphelpers.get_loggedinusers()
            for user in dict_user:
                if run_as_user_unix(user, ['/opt/wapt/wapt-get.bin', 'session-setup', 'ALL']):
                    nb = nb + 1
        self.progress = 100.0
        self.summary = _("session-setup run for %s session") % nb

    def same_action(self, other):
        # we launch only one session_setup
        return self.__class__ == other.__class__

    def __str__(self):
        return _("Run Session-Setup")

class WaptWUATask(WaptTask):

    def wua_search_progress_cb(self, progress: float, duration_secs: int, finished: bool) -> bool:
        # Finished
        if finished:
            if progress < 0:
                self.runstatus = 'Error scanning updates: Exception occured'
                self.progress = 100.0
            else:
                self.runstatus = 'Done searching'
                self.progress = progress
        # Init
        elif progress < 0:
            self.runstatus = 'Waiting for WUA search to complete'
        # Running
        else:
            # Stop requested
            if self.wapt and self.wapt.task_is_cancelled.is_set():
                return True
            self.progress = progress
            self.runstatus = 'Searching %ds' % duration_secs

    def install_callback(self, current_index, installs_count, current_progress, overall_progress):
        msg = _('Installing update %s / %s %s%% completed') % (current_index+1, installs_count, overall_progress)
        #print('py install_callback: self:%s , %s %s %s %s' % (self, current_index, installs_count, current_progress, overall_progress))
        self.progress = overall_progress
        self.runstatus = msg

        if not self.wapt:
            return
        if self.wapt.progress_hook is None:
            print(msg)
        else:
            cancel_request = self.wapt.progress_hook(True, msg, overall_progress, installs_count)
            if cancel_request:
                print(_('Cancel Windows updates install requested'))
                return True


class WaptWUAScanTask(WaptWUATask):
    """Launch Windows updates scan and save results in local wapt status db"""

    def __init__(self, **args):
        super(WaptWUAScanTask, self).__init__()
        self.notify_server_on_start = False
        self.notify_server_on_finish = True
        self.force = False
        self.uuids = []
        self.priority = 1000
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        check_battery()
        self.update_status(_("Start Windows Updates Scans"))
        self.progress = 0.0
        with self.wapt.waptwua(True) as wua:
            old_cb = wua.get_search_progress_callback()
            try:
                wua.set_search_progress_callback(self.wua_search_progress_cb)
                self.result = wua.scan_updates_status(self.force)
                self.summary = render_mustache(SUMMARY_TEMPLATE, wua.summary_status())
            finally:
                wua.set_search_progress_callback(old_cb)
        self.progress = 100.0

    def __str__(self):
        return _("Scan pending Windows updates")


class WaptWUADowloadTask(WaptWUATask):
    """Launch Windows updates scan and downloads missing updates in local wuauagent cache"""

    def __init__(self, **args):
        super(WaptWUADowloadTask, self).__init__()
        self.notify_server_on_start = False
        self.notify_server_on_finish = True
        self.force = False
        self.uuids = []
        self.priority = 1000
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        check_battery()
        self.update_status(_("Start Windows Updates Scan and download"))
        self.progress = 0.0
        with self.wapt.waptwua(True) as wua:
            old_cb = wua.get_search_progress_callback()
            try:
                wua.set_search_progress_callback(self.wua_search_progress_cb)
                print('Download %s updates' % (self.uuids or 'all'))
                self.result = wua.download_updates()
                self.summary = render_mustache(SUMMARY_TEMPLATE, wua.summary_status())
            finally:
                wua.set_search_progress_callback(old_cb)

        self.progress = 100.0

    def __str__(self):
        return _("Scan and download pending Windows updates")


class WaptWUAInstallTask(WaptWUATask):
    """Launch install of pending Windows updates"""

    def __init__(self, **args):
        super(WaptWUAInstallTask, self).__init__()
        self.notify_server_on_start = False
        self.notify_server_on_finish = True
        self.force = False
        self.uuids = None
        self.priority = 1000
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        check_battery()
        if self.uuids is not None:
            self.update_status(_("Start install of %s Windows Updates") % len(self.uuids))
        else:
            self.update_status(_("Start install of %s Windows Updates") % 'all pending')
        self.progress = 0.0
        with self.wapt.waptwua(True) as wua:
            old_cb = wua.get_search_progress_callback()
            try:
                wua.set_search_progress_callback(self.wua_search_progress_cb)
                wua.set_install_progress_callback(self.install_callback)

                self.result = wua.install_updates(self.force, update_ids=self.uuids)
                self.summary = render_mustache(SUMMARY_TEMPLATE, wua.summary_status())
            finally:
                wua.set_search_progress_callback(old_cb)
        self.progress = 100.0

    def __str__(self):
        return _("Install Windows updates")


class WaptWUAUninstallTask(WaptWUATask):
    """Launch install of pending Windows updates"""

    def __init__(self, **args):
        super(WaptWUAUninstallTask, self).__init__()
        self.notify_server_on_start = False
        self.notify_server_on_finish = True
        self.force = False
        self.uuids = []
        self.priority = 1000
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        check_battery()
        self.update_status(_("Start uninstall of %s Windows Updates") % len(self.uuids))
        self.progress = 0.0
        with self.wapt.waptwua(True) as wua:
            old_cb = wua.get_search_progress_callback()
            try:
                wua.set_search_progress_callback(self.wua_search_progress_cb)
                self.result = wua.uninstall_updates(update_ids=self.uuids)
                self.summary = render_mustache(SUMMARY_TEMPLATE, wua.summary_status())
            finally:
                wua.set_search_progress_callback(old_cb)
        self.progress = 100.0

    def __str__(self):
        return _("Uninstall Windows updates")


class WaptHostReboot(WaptTask):
    """A task to restart the waptservice using a spawned cmd process"""

    def __init__(self, **args):
        super(WaptHostReboot, self).__init__()
        self.priority = 10000
        self.notify_server_on_start = False
        self.notify_server_on_finish = False
        self.notify_user = False
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        """"""
        if platform.system() == 'Windows':
            args = ['/t 15', '/r']
            if self.force:
                args.append('/f')
            output = setuphelpers.run('"%s" %s' % (makepath(system32(), 'shutdown.exe'), ' '.join(args)))
        elif platform.system() == 'Darwin':
            output = setuphelpers.run('shutdown -r now')
        else:
            output = setuphelpers.run('shutdown -r now')
        self.update_status('Reboot triggered: %s' % output)
        self.result = {'result': 'OK', 'message': output}

    def __str__(self):
        return _("Rebooting host")


class WaptHostShutdown(WaptTask):
    """A task to Shutdown the host"""

    def __init__(self, **args):
        super(WaptHostShutdown, self).__init__()
        self.priority = 10000
        self.notify_server_on_start = False
        self.notify_server_on_finish = False
        self.notify_user = False
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        """"""
        if platform.system() == 'Windows':
            args = ['/t 15', '/s']
            if self.force:
                args.append('/f')
            output = setuphelpers.run('"%s" %s' % (makepath(system32(), 'shutdown.exe'), ' '.join(args)))
        elif platform.system() == 'Darwin':
            output = setuphelpers.run('shutdown -h now')
        else:
            output = setuphelpers.run('shutdown -h now')
        self.update_status('Shutdown triggered: %s' % output)
        self.result = {'result': 'OK', 'message': output}

    def __str__(self):
        return _("Shutdown host")


clean_mgr_topics = [
    'Active Setup Temp Folders',
    'Content Indexer Cleaner',
    'Downloaded Program Files',
    'GameNewsFiles',
    'GameStatisticsFiles',
    'GameUpdateFiles',
    'Internet Cache Files',
    'Memory Dump Files',
    'Offline Pages Files',
    'Old ChkDsk Files',
    'Previous Installations',
    'Recycle Bin',
    'Service Pack Cleanup',
    'Setup Log Files',
    'System error memory dump files',
    'System error minidump files',
    'Temporary Files',
    'Temporary Setup Files',
    'Temporary Sync Files',
    'Thumbnail Cache',
    'Update Cleanup',
    'Upgrade Discarded Files',
    'Windows Error Reporting Archive Files',
    'Windows Error Reporting Queue Files',
    'Windows Error Reporting System Archive Files',
    'Windows Error Reporting System Queue Files',
    'Windows Upgrade Log Files',
]


def run_cleanmgr():
    print('Clearing CleanMgr.exe automation settings 0099.')
    volume_cache = reg_openkey_noredir(HKEY_LOCAL_MACHINE, r'SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\VolumeCaches')
    for key in reg_enum_subkeys(volume_cache):
        print('Reset %s' % key)
        reg_delvalue(reg_openkey_noredir(volume_cache, key, KEY_ALL_ACCESS), 'StateFlags0099')

    for item in reg_enum_subkeys(volume_cache):
        if item in clean_mgr_topics:
            print('Enable cleanup of %s' % item)
            key = reg_openkey_noredir(HKEY_LOCAL_MACHINE, r'SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\VolumeCaches\%s' % item, KEY_ALL_ACCESS)
            reg_setvalue(key, 'StateFlags0099', 2, REG_DWORD)

    print('Starting CleanMgr.exe...')
    cleanmgr_path = os.path.expandvars(r'%SystemRoot%\System32\CleanMgr.exe')
    args = '/verylowdisk /sagerun:99'
    start_interactive_process(cleanmgr_path, args, hide=True)

    print('Waiting for CleanMgr and DismHost processes.')
    start = time.time()
    timeout = 120
    while setuphelpers.isrunning('cleanmgr') or setuphelpers.isrunning('dismhost'):
        if time.time() - start > timeout:
            raise Exception('Tiemout waiting for cleanmgr and dism host to terminate')
        time.sleep(0.2)

    print('Check result')
    logs_fn = os.path.expandvars(r'%SystemRoot%\Logs\CBS\DeepClean.log')
    if os.path.isfile(logs_fn):
        with open(os.path.expandvars(r'%SystemRoot%\Logs\CBS\DeepClean.log'), 'r') as f:
            logs = f.read()
        result = '\n'.join(re.findall(r'Total size of superseded packages:(.*)', logs))
        if result:
            print('Restart of computer is required to complete cleanup...')
            print('Result:\n%s' % result)
    else:
        result = 'No log file'

    return result


class WaptRunCleanMgr(WaptTask):
    """Launch CleanMgr on local host"""

    def __init__(self, **args):
        super(WaptRunCleanMgr, self).__init__()
        self.notify_server_on_start = False
        self.notify_server_on_finish = True
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        self.update_status(_("Start CleanMgr command"))
        self.progress = 0.0
        self.result = run_cleanmgr()
        self.progress = 100.0

    def same_action(self, other):
        return False

    def __str__(self):
        return _("Run CleanMgr")

def arg_is_true(arg='0'):
    if isinstance(arg,bool):
        return arg
    return str(arg).lower() in ('true','1')

def start_waptexit(sio, action, verified_by):
    result = []

    only_priorities = action.get('only_priorities', None)
    only_if_not_process_running = arg_is_true(action.get('only_if_not_process_running', 'True'))

    if sys.platform == 'win32':
        install_wua_updates = arg_is_true(action.get('install_wua_updates', 'False'))
    else:
        install_wua_updates = False

    countdown = action.get('waptexit_countdown', 300)
    allow_cancel_upgrade = arg_is_true(action.get('allow_cancel_upgrade', 'True'))

    args = ['/waptexit_countdown=%s' % countdown]

    if allow_cancel_upgrade:
        args.append('/allow_cancel_upgrade')

    if only_priorities:
        args.append('/priorities=%s' % only_priorities)

    if install_wua_updates:
        args.append('/install_wua_updates=%s' % install_wua_updates)

    args.append('/only_if_not_process_running=%s' % only_if_not_process_running)

    if sys.platform == 'win32':
        list_session = get_active_sessions()
        if len(list_session) > 0:
            for session_id in list_session:
                process_info = start_interactive_process(os.path.join(wapt_root_dir, 'waptexit.exe'), ' '.join(args), session_id=session_id)
                if process_info is not None:
                    result.append(process_info)
            return dict(result=len(result), summary='waptexit launched for %s sessions' % len(result))
    else:
        dict_user = setuphelpers.get_loggedinusers()
        if platform.system() == 'Darwin':
            path_to_waptexit = '/Applications/WAPT/WAPT Exit.app/Contents/MacOS/waptexit'
        else:
            path_to_waptexit = '/opt/wapt/waptexit.bin'
        if os.path.isfile(path_to_waptexit):
            for user in dict_user:
                if user == 'root' and sys.platform.startswith('darwin'):
                    continue
                cmd = [path_to_waptexit]
                cmd.extend(args)
                r = run_as_user_unix(user, cmd)
                if r:
                    result.append(user)
                if sys.platform.startswith('darwin'):
                    break
        return dict(result=bool(len(result)), summary=_('waptexit launched for %s sessions') % len(result))

    if not result and sio:
        sio.task_manager.add_task(WaptUpgrade(notify_user=False,
                                              created_by=verified_by,
                                              only_priorities=only_priorities,
                                              only_if_not_process_running=only_if_not_process_running,
                                              force=False
                                              )).as_dict()

        return dict(result=True, summary=_('no user log in, launch upgrade'))


def run_wol(sio, action, verified_by):
    if  waptconfig.wol_relay:
        macs = action.get('macs')
        address = '255.255.255.255'
        for mac in macs:
            if not isinstance(mac,list):
                mac = [mac]
            try:
                for port in waptconfig.wol_port.split(','):
                    logger.debug('Send wol at address %s mac %s port %s verified_by %s' % (address, mac, port, verified_by))
                    wakeonlan.send_magic_packet(
                        * mac,
                        ip_address=address,
                        port=int(port.strip()))

                    for broadcast in action.get('broadcast'):
                        logger.debug('Send wol at address %s mac %s port %s verified_by %s' % (broadcast, mac, port, verified_by))
                        wakeonlan.send_magic_packet(
                            * mac,
                            ip_address=broadcast,
                            port=int(port.strip()))
            except ValueError :
                pass


        return dict(result=True, summary=_('Send wol at address %s mac %s port %s verified_by %s') % (address, macs, port, verified_by))


def show_message(sio, action, verified_by):
    amsg = action['msg'].replace('\\n', '\n')
    amsg = '%s' % (amsg)
##    amsg = amsg.encode('utf8')
##    amsg_b64 = base64.b64encode(amsg)
##    amsg_b64 = amsg_b64.decode()
    try:
        display_time = action['display_time']
        display_time = '%s' % (display_time)
    except:
        display_time = '0'

    uuid_message= str(uuid.uuid4())
    if not os.path.isdir(makepath(wapt_root_dir,'cache')):
        os.makedirs(makepath(wapt_root_dir,'cache'))
    path_message = makepath(wapt_root_dir,'cache',uuid_message)

    with codecs.open(path_message,'w',encoding='utf8') as f:
        f.write(amsg)

    if sys.platform == 'win32':
        waptmessage_path = os.path.join(wapt_root_dir, 'waptmessage.exe')
        for session_id in get_active_sessions():
            start_interactive_process(waptmessage_path, '-f "%s" -t %s' % (path_message, display_time), session_id=session_id)
        time.sleep(30)
        setuphelpers.remove_file(path_message)
        return dict(result=True, summary=_('Message displayed'))
    else:
        result = []
        if sys.platform == 'darwin':
            waptmessage_path = '/Applications/WAPT/WAPT Message.app/Contents/MacOS/waptmessage'
        else:
            waptmessage_path = os.path.join(wapt_root_dir, 'waptmessage.bin')
        if os.path.isfile(waptmessage_path):
            for user in setuphelpers.get_loggedinusers():
                result.append(run_as_user_unix(user, [waptmessage_path, '-f', path_message, '-t', display_time]))
        time.sleep(30)
        setuphelpers.remove_file(path_message)
        return dict(result=bool(len(result)), summary=_('Message displayed for %s sessions') % len(result))


def trigger_gpupdate(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    task = WaptGPUpdate(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, force=force, created_by=verified_by)
    result = sio.task_manager.add_task(task)
    return result.as_dict()


def trigger_waptwua_scan(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    task = WaptWUAScanTask(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, created_by=verified_by, force=force)
    result = sio.task_manager.add_task(task)
    return [result.as_dict()]


def trigger_waptwua_download(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    uuids = action.get('uuids', None)
    task = WaptWUADowloadTask(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, created_by=verified_by, force=force, uuids=uuids)
    result = sio.task_manager.add_task(task)
    return [result.as_dict()]


def trigger_waptwua_install(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    uuids = action.get('uuids', None)
    task = WaptWUAInstallTask(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, created_by=verified_by, force=force, uuids=uuids)
    result = sio.task_manager.add_task(task)
    return [result.as_dict()]


def trigger_waptwua_uninstall(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    uuids = action.get('uuids', None)
    task = WaptWUAUninstallTask(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, created_by=verified_by, force=force, uuids=uuids)
    result = sio.task_manager.add_task(task)
    return [result.as_dict()]


def trigger_host_audit(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    now = setuphelpers.currentdatetime()

    result = []
    packagenames = []
    for package_status in sio.wapt.installed():
        if force or not package_status.next_audit_on or (now >= package_status.next_audit_on):
            packagenames.append(package_status.package)

    if packagenames:
        task = WaptAuditPackage(packagenames=packagenames, notify_user=notify_user, force=force, created_by=verified_by)
        result.append(sio.task_manager.add_task(task).as_dict())
    if notify_server_on_finish:
        result.append(sio.task_manager.add_task(WaptUpdateServerStatus(priority=100, created_by=verified_by)).as_dict())
    return result


def trigger_audit_packages(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    now = setuphelpers.currentdatetime()
    packagenames = ensure_list(action.get('packages', None), allow_none=True)

    result = []
    for package_status in sio.wapt.installed():
        if not package_status.package in packagenames:
            continue
        if force or not package_status.next_audit_on or (now >= package_status.next_audit_on):
            task = WaptAuditPackage(packagenames=package_status.package, notify_user=notify_user, force=force, created_by=verified_by)
            result.append(sio.task_manager.add_task(task).as_dict())
    if notify_server_on_finish:
        result.append(sio.task_manager.add_task(WaptUpdateServerStatus(priority=100, created_by=verified_by)).as_dict())
    return result


def trigger_cleanmgr(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    result = []
    task = WaptRunCleanMgr(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, force=force, created_by=verified_by)
    result.append(sio.task_manager.add_task(task))
    return result


def trigger_host_shutdown(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    only_if_no_user_session = action.get('only_if_no_user_session', False)

    result = []
    if sio.config.allow_remote_shutdown:
        result.append(sio.task_manager.add_task(WaptHostShutdown(notify_user=notify_user,
                                                                 created_by=verified_by,
                                                                 only_if_no_user_session=only_if_no_user_session,
                                                                 notify_server_on_finish=notify_server_on_finish,
                                                                 force=force
                                                                 )).as_dict())


def trigger_host_reboot(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)
    only_if_no_user_session = action.get('only_if_no_user_session', False)

    result = []
    if sio.config.allow_remote_reboot:
        result.append(sio.task_manager.add_task(WaptHostReboot(notify_user=notify_user,
                                                               notify_server_on_finish=notify_server_on_finish,
                                                               created_by=verified_by,
                                                               only_if_no_user_session=only_if_no_user_session,
                                                               force=force
                                                               )).as_dict())


def run_scheduled_wua_scan(waptservice, force=False):
    with waptservice.wapt.waptwua(True) as client:
        client.scan_updates_status(force=force)


def run_scheduled_wua_downloads(waptservice, force=False):
    with waptservice.wapt.waptwua(True) as client:
        client.download_updates()


def run_scheduled_wua_installs(waptservice, force=False):
    with waptservice.wapt.waptwua(True) as client:
        if force or client.stored_waptwua_status.get('status') != 'OK':
            client.install_updates(force=force)


def trigger_session_setup(sio, action, verified_by):
    notify_user = action.get('notify_user', False)
    notify_server_on_finish = action.get('notify_server', False)
    force = action.get('force', False)

    result = []
    if setuphelpers.get_loggedinusers():
        result.append(sio.task_manager.add_task(WaptRunSessionSetup(notify_user=notify_user,
                                                                notify_server_on_finish=notify_server_on_finish,
                                                                created_by=verified_by,
                                                                force=force,
                                                                )).as_dict())


register_remote_action('start_waptexit', start_waptexit, [])
register_remote_action('show_message', show_message, ['msg'])
register_remote_action('trigger_gpupdate', trigger_gpupdate, [])
register_remote_action('trigger_waptwua_scan', trigger_waptwua_scan, [])
register_remote_action('trigger_waptwua_download', trigger_waptwua_download, [])
register_remote_action('trigger_waptwua_install', trigger_waptwua_install, [])
register_remote_action('trigger_waptwua_uninstall', trigger_waptwua_uninstall, [])
register_remote_action('trigger_host_audit', trigger_host_audit, [])
register_remote_action('trigger_audit_packages', trigger_audit_packages, [])
register_remote_action('trigger_cleanmgr', trigger_cleanmgr, [])
register_remote_action('trigger_host_reboot', trigger_host_reboot, [])
register_remote_action('trigger_host_shutdown', trigger_host_shutdown, [])
register_remote_action('trigger_session_setup', trigger_session_setup, [])
register_remote_action('run_wol', run_wol, [])

# WAPT WUA http actions


@waptwua_api.route('/trigger_waptwua_scan')
@requires_enterprise
@allow_local_token
def enqueue_waptwua_scan():
    force = int(request.args.get('force', '0')) != 0
    notify_user = int(request.args.get('notify_user', '0' if not waptconfig.notify_user else '1')) != 0
    notify_server_on_finish = int(request.args.get('notify_server', '0')) != 0

    task = WaptWUAScanTask(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, force=force)
    data = waptwua_api.task_manager.add_task(task)

    if request.args.get('format', 'html') == 'json' or request.path.endswith('.json'):
        return Response(jsondump(data), mimetype='application/json')
    else:
        return render_wapt_template('install.html', data=data)


@waptwua_api.route('/trigger_waptwua_download')
@requires_enterprise
@allow_local_token
def enqueue_waptwua_download():
    force = int(request.args.get('force', '0')) != 0
    notify_user = int(request.args.get('notify_user', '0' if not waptconfig.notify_user else '1')) != 0
    notify_server_on_finish = int(request.args.get('notify_server', '0')) != 0
    uuids = request.args.get('uuids', None)
    if uuids:
        uuids = ensure_list(uuids)

    task = WaptWUADowloadTask(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, force=force, uuids=uuids)
    data = waptwua_api.task_manager.add_task(task)

    if request.args.get('format', 'html') == 'json' or request.path.endswith('.json'):
        return Response(jsondump(data), mimetype='application/json')
    else:
        return render_wapt_template('install.html', data=data)


@waptwua_api.route('/trigger_waptwua_install')
@requires_enterprise
@allow_local_token
def enqueue_waptwua_install():
    force = int(request.args.get('force', '0')) != 0
    notify_user = int(request.args.get('notify_user', '0' if not waptconfig.notify_user else '1')) != 0
    notify_server_on_finish = int(request.args.get('notify_server', '0')) != 0
    uuids = request.args.get('uuids', None)
    if uuids:
        uuids = ensure_list(uuids)

    task = WaptWUAInstallTask(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, force=force, uuids=uuids)
    data = waptwua_api.task_manager.add_task(task)

    if request.args.get('format', 'html') == 'json' or request.path.endswith('.json'):
        return Response(jsondump(data), mimetype='application/json')
    else:
        return render_wapt_template('install.html', data=data)


@waptwua_api.route('/trigger_waptwua_uninstall')
@requires_enterprise
@allow_local_token
def enqueue_waptwua_uninstall():
    force = int(request.args.get('force', '0')) != 0
    notify_user = int(request.args.get('notify_user', '0' if not waptconfig.notify_user else '1')) != 0
    notify_server_on_finish = int(request.args.get('notify_server', '0')) != 0
    uuids = request.args.get('uuids', None)
    if uuids:
        uuids = ensure_list(uuids)

    task = WaptWUAUninstallTask(notify_user=notify_user, notify_server_on_finish=notify_server_on_finish, force=force, uuids=uuids)
    data = waptwua_api.task_manager.add_task(task)

    if request.args.get('format', 'html') == 'json' or request.path.endswith('.json'):
        return Response(jsondump(data), mimetype='application/json')
    else:
        return render_wapt_template('install.html', data=data)


def give_display_number_unix(username=None):
    """ Returns the ID of a UNIX display, given the username whom it belongs to"""
    display_number = None

    # macOS : useless, since it should only ever draw on the one active display, but seems doable
    # Quartz Compositor (the macOS display server) seems to be hard to interact with
    # https://developer.apple.com/documentation/coregraphics/quartz_display_services
    if sys.platform.startswith('darwin'):
        pass  # import Quartz ; Quartz.CGMainDisplayID()
        return ":0"

    if sys.platform.startswith('linux'):
        # Fetching the display ID from the X/Xwayland command arguments
        display_number = get_x_session_id_by_user(username)

        # Fetching the display ID from the environment
        if not display_number:
            for i in glob.glob('/proc/*/environ'):
                if os.stat(i).st_uid == getpwnam(username).pw_uid:
                    with open(i, 'r') as f:
                        data = f.read()
                        if 'DISPLAY=' in data:
                            return data.split('DISPLAY=')[1].split('\x00')[0].strip('localhost').split('.')[0]

        if not display_number:
            logger.error('Could not get a display ID for Linux.')
        return display_number

    logger.error('Platform not supported : could not get a display ID')
    return None


def get_x_session_id_by_user(username):
    """ Returns the ID of an active X or XWayland session, given the username whom it belongs to"""
    x_processes_names = ['xorg', 'xwayland']

    for x_ps_name in x_processes_names:
        processes = setuphelpers.get_processes_with_name(x_ps_name)
        if not processes:
            continue

        # Find the mention of Xorg/XWayland in the command, then take its argument (the session ID)

        sessions = []
        for ps in processes:
            try:
                session = {}
                session['user'] = ps['username']
                session['X_ID'] = ps['cmdline'][1]
                sessions.append(session)
            except:
                logger.error('Could not find a display ID in {}'.format(ps))
                continue

        user_sessions = [d['X_ID'] for d in sessions if d['user'] == username]
        if not len(user_sessions):
            continue

        # current behavior : assume user only has one X session
        return user_sessions[0]
    return None


def run_as_user_unix(username, cmd):
    display_number = give_display_number_unix(username)
    if display_number and display_number.startswith(':'):
        uiduser = getpwnam(username).pw_uid
        giduser = getpwnam(username).pw_uid
        homedir = getpwnam(username).pw_dir
        subprocess.Popen(cmd, preexec_fn=demote(uiduser, giduser), env=dict(DISPLAY=display_number, PATH="/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin", HOME=homedir,DBUS_SESSION_BUS_ADDRESS="unix:path=/run/user/%s/bus" % uiduser,XDG_RUNTIME_DIR="/run/user/%s" % uiduser))
        return username


def demote(user_uid, user_gid):
    """Pass the function 'set_ids' to preexec_fn, rather than just calling
    setuid and setgid. This will change the ids for that subprocess only"""

    def set_ids():
        os.setgid(user_gid)
        os.setuid(user_uid)

    return set_ids


