#!/usr/bin/env python3
##
## -----------------------------------------------------------------
##    This file is part of WAPT Software Deployment
##    Copyright (C) 2012 - 2024  Tranquil IT https://www.tranquil.it
##    All Rights Reserved.
##
##    WAPT helps systems administrators to efficiently deploy
##    setup, update and configure applications.
## ------------------------------------------------------------------
##


import os
import shutil
import json
import time
import urllib.parse
import urllib.request
import urllib.parse
import urllib.error
import traceback
import logging

from waptservice.waptservice_common import WaptTask, waptconfig, _

from waptutils import wget, get_sha256

if os.name == 'nt':
    NGINX_GID = None
else:
    try:
        import grp
        import pwd
        from setuphelpers_linux import type_debian, type_redhat
        WAPT_UID = pwd.getpwnam('root').pw_uid
        if type_debian():
            NGINX_GID = grp.getgrnam('www-data').gr_gid
        elif type_redhat():
            NGINX_GID = grp.getgrnam('nginx').gr_gid
        else:
            NGINX_GID = None
    except:
        NGINX_GID = None

logger = logging.getLogger('repositories')

class WaptSyncRepo(WaptTask):
    """Task for sync the server Repo in the current machine"""

    def __init__(self, audit=False, socketio_client=None, **args):
        super(WaptSyncRepo, self).__init__()
        self.local_repo_path = waptconfig.local_repo_path
        self.socketio_client = socketio_client
        self.speed = waptconfig.local_repo_limit_bandwidth
        self.totalsize = 0
        self.nb_elem_avg = 0
        self.last_avg = 0.0
        self.audit = audit
        self.totaldownloaded = 0
        self.lastprogressdownload = 0
        self.timefordownload = 0
        self.countchunk = 0
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        """Launch the sync of the server"""

        self.progress = 0.0

        def dict_to_delete_rec(pathtosync, dict_to_delete, pathfile):
            for afile in dict_to_delete[pathfile]['files'].keys():
                new_pathfile = os.path.normpath(os.path.join(pathtosync, afile))
                dict_to_delete[new_pathfile] = dict(dict_to_delete[pathfile]['files'][afile])
                if dict_to_delete[new_pathfile]['isDir']:
                    dict_to_delete_rec(pathtosync, dict_to_delete[new_pathfile]['files'], new_pathfile)
                    del dict_to_delete[new_pathfile]['files']

        def read_tree_of_files_and_init(srvname='', pathtosync='', atree={}, sync_dict={}, errors={}, result={}, treedepth=0, dict_diff=None):
            list_removable_sync_dict = list(sync_dict.keys())
            if not(result):
                result = {}
                result['errors'] = {}
                result['modified'] = {}
                result['new'] = {}
                result['deleted'] = {}
                result['success'] = True
                result['todownload'] = {}
                result['dict_to_delete'] = {}
                result['total_to_download'] = 0
            for afile in atree.keys():
                if not(treedepth) and (afile not in waptconfig.remote_repo_dirs):
                    continue
                if treedepth == 1 and dict_diff is not None:
                    parent_dir = os.path.dirname(afile)
                    if not(os.path.basename(os.path.normpath(afile)) in dict_diff[parent_dir]):
                        continue
                pathfile = os.path.normpath(os.path.join(pathtosync, afile))
                if afile in errors:
                    try:
                        del sync_dict[afile]
                        del errors[afile]
                        list_removable_sync_dict.remove(afile)
                        if sync_dict[afile]['isDir']:
                            shutil.rmtree(pathfile, ignore_errors=True)
                        else:
                            os.remove(pathfile)
                    except:
                        pass
                try:
                    if (sync_dict.get(afile)):
                        list_removable_sync_dict.remove(afile)
                        if atree[afile]['isDir'] and sync_dict[afile]['isDir']:
                            read_tree_of_files_and_init(srvname, pathtosync, atree[afile]['files'], sync_dict[afile]['files'], errors, result, treedepth+1, dict_diff)
                        elif not(atree[afile]['isDir']) and not(sync_dict[afile]['isDir']):
                            if sync_dict[afile]['sum'] == atree[afile]['sum']:
                                continue
                            else:
                                url = urllib.parse.urljoin(srvname, urllib.request.pathname2url(afile))
                                result['todownload'][afile] = {}
                                result['todownload'][afile]['url'] = url
                                result['todownload'][afile]['sum'] = atree[afile]['sum']
                                result['todownload'][afile]['path'] = pathfile
                                result['total_to_download'] += atree[afile]['size']
                                sync_dict[afile]['isDir'] = False
                                sync_dict[afile]['sum'] = atree[afile]['sum']
                                sync_dict[afile]['size'] = atree[afile]['size']
                                result['modified'][afile] = url
                        else:
                            if atree[afile]['isDir']:
                                os.remove(pathfile)
                                sync_dict[afile]['isDir'] = True
                                sync_dict[afile]['files'] = {}
                                del sync_dict[afile]['sum']
                                result['modified'][afile] = 'toDir'
                                read_tree_of_files_and_init(srvname, pathtosync, atree[afile]['files'], sync_dict[afile]['files'], errors, result, treedepth+1, dict_diff)
                            else:
                                shutil.rmtree(pathfile, ignore_errors=True)
                                sync_dict[afile]['isDir'] = False
                                sync_dict[afile]['sum'] = atree[afile]['sum']
                                del sync_dict[afile]['files']
                                url = urllib.parse.urljoin(srvname, urllib.request.pathname2url(afile))
                                result['todownload'][afile] = {}
                                result['todownload'][afile]['url'] = url
                                result['todownload'][afile]['sum'] = atree[afile]['sum']
                                result['todownload'][afile]['path'] = pathfile
                                result['total_to_download'] += atree[afile]['size']
                                sync_dict[afile]['isDir'] = False
                                sync_dict[afile]['sum'] = atree[afile]['sum']
                                sync_dict[afile]['size'] = atree[afile]['size']
                                result['modified'][afile] = url
                    else:
                        sync_dict[afile] = {}
                        if atree[afile]['isDir']:
                            sync_dict[afile]['isDir'] = True
                            sync_dict[afile]['files'] = {}
                            if not os.path.exists(pathfile):
                                os.mkdir(pathfile)
                            if NGINX_GID is not None:
                                os.chown(pathfile, WAPT_UID, NGINX_GID)
                            result['new'][afile] = 'isDir'
                            read_tree_of_files_and_init(srvname, pathtosync, atree[afile]['files'], sync_dict[afile]['files'], errors, result, treedepth+1, dict_diff)
                        else:
                            sync_dict[afile]['isDir'] = False
                            sync_dict[afile]['sum'] = atree[afile]['sum']
                            sync_dict[afile]['size'] = atree[afile]['size']
                            if os.path.isfile(pathfile):
                                if get_sha256(pathfile) == atree[afile]['sum']:
                                    if NGINX_GID is not None:
                                        os.chown(pathfile, WAPT_UID, NGINX_GID)
                                    continue
                                else:
                                    os.unlink(pathfile)
                            url = urllib.parse.urljoin(srvname, urllib.request.pathname2url(afile))
                            result['todownload'][afile] = {}
                            result['todownload'][afile]['url'] = url
                            result['todownload'][afile]['sum'] = atree[afile]['sum']
                            result['todownload'][afile]['path'] = pathfile
                            result['total_to_download'] += atree[afile]['size']
                            result['new'][afile] = url
                except Exception:
                    result['errors'][afile] = traceback.format_exc()
                    result['success'] = False
            for afile in list_removable_sync_dict:
                pathfile = os.path.normpath(os.path.join(pathtosync, afile))
                result['dict_to_delete'][pathfile] = dict(sync_dict[afile])
                if sync_dict[afile]['isDir']:
                    dict_to_delete_rec(pathtosync, result['dict_to_delete'], pathfile)
                    del result['dict_to_delete'][pathfile]['files']
                del sync_dict[afile]
            return result

        def audit_tree_of_files(pathtosync='', atree={}, result={}, treedepth=0, dict_diff=None, number_of_files=0):
            if not(result):
                result = {}
                result['actual_number_of_files'] = number_of_files
                result['errors'] = {}
            if NGINX_GID is not None:
                rights_to_check = []
            for afile in atree.keys():
                try:
                    if not(treedepth) and (afile not in waptconfig.remote_repo_dirs):
                        continue
                    if treedepth == 1 and dict_diff is not None:
                        parent_dir = os.path.dirname(afile)
                        if not(os.path.basename(os.path.normpath(afile)) in dict_diff[parent_dir]):
                            continue
                    pathfile = os.path.normpath(os.path.join(pathtosync, afile))
                    if atree[afile]['isDir']:
                        if os.path.isdir(pathfile):
                            if NGINX_GID is not None:
                                rights_to_check.append(pathfile)
                            audit_tree_of_files(pathtosync, atree[afile]['files'], result, treedepth+1, dict_diff, number_of_files)
                        else:
                            result['errors'][afile] = 'Missing directory'
                    else:
                        if os.path.isfile(pathfile):
                            if atree[afile]['sum'] != get_sha256(pathfile):
                                result['errors'][afile] = 'Wrong sum file modified'
                            elif NGINX_GID is not None:
                                rights_to_check.append(pathfile)
                        else:
                            result['errors'][afile] = 'Missing file'
                        result['actual_number_of_files'] -= 1
                        self.progress = ((number_of_files-result['actual_number_of_files'])*100)/float(number_of_files)
                        if self.socketio_client and (not((number_of_files-result['actual_number_of_files']) % 5) or (number_of_files-result['actual_number_of_files'] == 0)):
                            d = {'progress': self.progress}
                            self.socketio_client.emit('sync_progress', d)
                except Exception:
                    result['errors'][afile] = traceback.format_exc()
            if NGINX_GID is not None:
                for afile in rights_to_check:
                    stat_info = os.stat(afile)
                    if stat_info.st_uid != WAPT_UID or stat_info.st_gid != NGINX_GID:
                        os.chown(afile, WAPT_UID, NGINX_GID)
            return result

        def count_files(atree={}):
            count = 0
            for afile in atree.keys():
                count += count_files(atree[afile]['files']) if atree[afile]['isDir'] else 1
            return count

        def download_files(dict_result={}, speed=None):
            def setter_progress(received, total, speed, url):
                self.totaldownloaded += -self.lastprogressdownload+received
                self.progress = (self.totaldownloaded/float(dict_result['total_to_download']))*100
                self.lastprogressdownload = received
                self.countchunk_current_file += 1
                if (self.countchunk_current_file > 1) and (received < total):
                    self.last_avg = ((self.last_avg*self.nb_elem_avg)+speed)/(self.nb_elem_avg+1)
                    self.nb_elem_avg += 1
                self.countchunk += 1
                if self.socketio_client and (self.countchunk_current_file == 1 or not(self.countchunk % 5)):
                    d = {
                        'progress': self.progress,
                        'current_download': url,
                        'current_speed': speed,
                    }
                    if self.last_avg != 0.0:
                        d['speed_average'] = self.last_avg
                    self.socketio_client.emit('sync_progress', d)

            for afile in dict_result['todownload']:
                try:
                    self.countchunk_current_file = 0
                    logger.info('Downloading : %s' % dict_result['todownload'][afile])
                    self.lastprogressdownload = 0
                    self.start_time_download = time.time()
                    wget(dict_result['todownload'][afile]['url'], target=dict_result['todownload'][afile]['path'], printhook=setter_progress, sha256=dict_result['todownload'][afile]['sum'], limit_bandwidth=speed, cert=self.wapt.waptserver.client_auth(), proxies=self.wapt.waptserver.proxies, verify_cert=self.wapt.waptserver.verify_cert)
                    if NGINX_GID is not None:
                        os.chown(dict_result['todownload'][afile]['path'], WAPT_UID, NGINX_GID)
                    logger.info('Downloaded in %f seconds.\n' % (time.time()-self.start_time_download))
                except Exception:
                    dict_result['errors'][afile] = traceback.format_exc()
                    dict_result['success'] = False
            logger.info('\n errors : %s' % dict_result['errors'])
            logger.info('\n success : %s' % dict_result['success'])
            del dict_result['todownload']

        def check_renamed_and_remove(dict_result={}):
            list_downloaded = []
            for afile_to_download in dict_result['todownload'].keys():
                for afile_to_remove in dict_result['dict_to_delete'].keys():
                    try:
                        if dict_result['dict_to_delete'][afile_to_remove]['isDir']:
                            continue
                        if dict_result['todownload'][afile_to_download]['sum'] == dict_result['dict_to_delete'][afile_to_remove]['sum']:
                            shutil.copyfile(afile_to_remove, dict_result['todownload'][afile_to_download]['path'])
                            dict_result['total_to_download'] -= dict_result['dict_to_delete'][afile_to_remove]['size']
                            result['total_to_download'] -= 1
                            list_downloaded.append(afile_to_download)
                            break
                    except Exception:
                        result['errors'][afile_to_download] = traceback.format_exc()
                        result['success'] = False
            for afile_downloaded in list_downloaded:
                del dict_result['todownload'][afile_downloaded]
            list_removed = []
            for afile_to_remove in dict_result['dict_to_delete'].keys():
                try:
                    if dict_result['dict_to_delete'][afile_to_remove]['isDir']:
                        result['deleted'][afile_to_remove] = 'isDir'
                        shutil.rmtree(afile_to_remove, ignore_errors=True)
                    else:
                        try:
                            result['deleted'][afile_to_remove] = 'isFile'
                            os.remove(afile_to_remove)
                        except Exception:
                            pass
                    list_removed.append(afile_to_remove)
                except Exception:
                    result['errors'][afile_to_remove] = traceback.format_exc()
                    result['success'] = False
            for afile_removed in list_removed:
                del dict_result['dict_to_delete'][afile_removed]

        filesync = os.path.join(self.local_repo_path, 'sync.json')
        if self.audit:
            try:
                dict_sync = self.wapt.waptserver.get('sync.json')
            except:
                logger.error('Unable to get sync.json file on server %s' % (self.wapt.waptserver.server_url))
                dict_sync = None
            if waptconfig.enable_diff_repo:
                try:
                    dict_diff_sync = self.wapt.waptserver.get('wapt-diff-repos/'+self.wapt.host_uuid+'.json')
                except:
                    logger.errror('Unable to get %s file on server %s' % ('wapt-diff-repos/'+self.wapt.host_uuid+'.json', self.wapt.waptserver.server_url))
                    dict_diff_sync = {'files': {'wapt': {'manual_add': [], 'auto_add': ['waptsetup.exe', 'waptagent.exe', 'waptdeploy.exe', 'Packages']}, 'waptwua': {'manual_add': [], 'auto_add': ['wsusscn2.cab']}, 'wapt-host': {'manual_add': [], 'auto_add': []}}, 'version': 0}
                dict_diff_sync['files']['wapt'] = dict_diff_sync['files']['wapt']['manual_add']+dict_diff_sync['files']['wapt']['auto_add']
                dict_diff_sync['files']['waptwua'] = dict_diff_sync['files']['waptwua']['manual_add']+dict_diff_sync['files']['waptwua']['auto_add']
                dict_diff_sync['files']['wapt-host'] = dict_diff_sync['files']['wapt-host']['manual_add']+dict_diff_sync['files']['wapt-host']['auto_add']
            else:
                dict_diff_sync = None
            if dict_sync:
                sync_version = {'version': dict_sync['version'], 'id': dict_sync['id']}
                if self.socketio_client:
                    self.socketio_client.emit('synchronization_audit_started')
                if os.path.isfile(filesync):
                    number_of_files = count_files(dict_sync['files'])
                    errors = audit_tree_of_files(self.local_repo_path, dict_sync['files'], dict_diff=dict_diff_sync['files'] if dict_diff_sync is not None else None, number_of_files=number_of_files)
                    del errors['actual_number_of_files']
                    if self.socketio_client:
                        if errors['errors']:
                            errors['version'] = sync_version
                            self.socketio_client.emit('synchronization_error', errors)
                        else:
                            self.socketio_client.emit('synchronization_audit_finished')
                    if errors['errors']:
                        filesync = os.path.join(self.local_repo_path, 'sync.json')
                        with open(filesync, 'r+') as f:
                            sync_dict = json.load(f)
                            sync_dict['errors'] = errors['errors']
                            f.seek(0, 0)
                            json.dump(sync_dict, f, indent=4)
                    result = errors
                else:
                    errors = {}
                    errors['version'] = sync_version
                    errors['errors'] = {}
                    errors['errors']['sync.json'] = 'No sync.json file relaunch sync'
                    if self.socketio_client:
                        self.socketio_client.emit('synchronization_error', errors)
        else:
            uptodate = True
            try:
                requests_head = self.wapt.waptserver.head('sync.json')
            except:
                requests_head = None

            if waptconfig.enable_diff_repo:
                try:
                    requests_head_diff_repo = self.wapt.waptserver.head('wapt-diff-repos/'+self.wapt.host_uuid+'.json')
                except:
                    requests_head_diff_repo = None
            else:
                requests_head_diff_repo = None
            result = {}
            if requests_head:
                if os.path.isdir(self.local_repo_path):
                    if os.path.isfile(filesync):
                        try:
                            with open(filesync, 'r') as f:
                                sync_dict = json.load(f)
                                if sync_dict['last-modified'] == requests_head['last-modified'] and sync_dict['sync-dirs'] == waptconfig.remote_repo_dirs:
                                    if requests_head_diff_repo is not None and requests_head_diff_repo['last-modified'] != sync_dict['last-modified_diff_repo']:
                                        uptodate = False
                                    if sync_dict['errors']:
                                        uptodate = False
                                    elif self.socketio_client:
                                        logger.info('\nAlready up-to-date with no errors')
                                        sync_version = {'version': sync_dict['version'], 'id': sync_dict['id']}
                                        self.socketio_client.emit('synchronization_finished', sync_version)
                                else:
                                    uptodate = False
                        except:
                            sync_dict = {}
                            uptodate = False
                    else:
                        sync_dict = {}
                        uptodate = False
                else:
                    sync_dict = {}
                    uptodate = False
                    os.mkdir(self.local_repo_path)
                    if NGINX_GID is not None:
                        os.chown(self.local_repo_path, WAPT_UID, NGINX_GID)
            else:
                logger.error("\nUnable to get sync.json file")

            if not(uptodate) and requests_head:
                try:
                    dict_sync = self.wapt.waptserver.get('sync.json')
                except:
                    dict_sync = None
                if waptconfig.enable_diff_repo:
                    try:
                        dict_diff_sync = self.wapt.waptserver.get('wapt-diff-repos/'+self.wapt.host_uuid+'.json')
                    except:
                        logger.error('Unable to get %s file on server %s' % ('wapt-diff-repos/'+self.wapt.host_uuid+'.json', self.wapt.waptserver.server_url))
                        dict_diff_sync = {'files': {'wapt': {'manual_add': [], 'auto_add': ['waptsetup.exe', 'waptagent.exe', 'waptdeploy.exe', 'Packages']}, 'waptwua': {'manual_add': [], 'auto_add': ['wsusscn2.cab']}, 'wapt-host': {'manual_add': [], 'auto_add': []}}, 'version': 0}
                    dict_diff_sync['files']['wapt'] = dict_diff_sync['files']['wapt']['manual_add']+dict_diff_sync['files']['wapt']['auto_add']
                    dict_diff_sync['files']['waptwua'] = dict_diff_sync['files']['waptwua']['manual_add']+dict_diff_sync['files']['waptwua']['auto_add']
                    dict_diff_sync['files']['wapt-host'] = dict_diff_sync['files']['wapt-host']['manual_add']+dict_diff_sync['files']['wapt-host']['auto_add']
                else:
                    dict_diff_sync = None
                if dict_sync:
                    sync_version = {'version': dict_sync['version'], 'id': dict_sync['id']}
                    if self.socketio_client:
                        self.socketio_client.emit('synchronization_started', sync_version)
                    if sync_dict:
                        if sync_dict['id'] != dict_sync['id']:
                            if not(os.path.isdir(self.local_repo_path)):
                                os.mkdir(self.local_repo_path)
                                if NGINX_GID is not None:
                                    os.chown(self.local_repo_path, WAPT_UID, NGINX_GID)
                            sync_dict = {}
                            sync_dict['files'] = {}
                            sync_dict['errors'] = {}
                    else:
                        sync_dict['files'] = {}
                        sync_dict['errors'] = {}
                    server_url = str(self.wapt.waptserver.server_url)
                    server_url = server_url if server_url.endswith('/') else server_url+'/'
                    result = read_tree_of_files_and_init(server_url, self.local_repo_path,dict_sync['files'], sync_dict['files'],sync_dict['errors'], dict_diff=dict_diff_sync['files'] if dict_diff_sync is not None else None)
                    logger.info('\ndeleted : %s' % result['deleted'])
                    logger.info('\nnew : %s' % result['new'])
                    logger.info('\nmodified : %s' % result['modified'])
                    if result['dict_to_delete']:
                        check_renamed_and_remove(result)
                    if result['todownload']:
                        logger.info('\nfiles to download : %s' % result['todownload'])
                        download_files(result, self.speed)
                    sync_dict['id'] = dict_sync['id']
                    sync_dict['version'] = dict_sync['version']
                    sync_dict['date'] = requests_head['date']
                    sync_dict['last-modified'] = requests_head['last-modified']
                    sync_dict['sync-dirs'] = waptconfig.remote_repo_dirs
                    if self.socketio_client:
                        if (result['success']):
                            self.socketio_client.emit('synchronization_finished', sync_version)
                        else:
                            errors = {}
                            errors['version'] = sync_version
                            errors['errors'] = result['errors']
                            self.socketio_client.emit('synchronization_error', errors)
                    sync_dict['success'] = result['success']
                    sync_dict['errors'] = result['errors']
                    if dict_diff_sync:
                        sync_dict['diff_sync_version'] = dict_diff_sync['version']
                        if requests_head_diff_repo is not None:
                            sync_dict['last-modified_diff_repo'] = requests_head_diff_repo['last-modified']
                    with open(filesync, 'w+') as f:
                        json.dump(sync_dict, f, indent=4)
                else:
                    logger.error("\nUnable to get sync.json file from server")

        self.result = result
        self.progress = 100.0

    def __str__(self):
        return _("Synchronizing server packages and WUA")
