#!/usr/bin/env python3
##
## -----------------------------------------------------------------
##    This file is part of WAPT Software Deployment
##    Copyright (C) 2012 - 2024  Tranquil IT https://www.tranquil.it
##    All Rights Reserved.
##
##    WAPT helps systems administrators to efficiently deploy
##    setup, update and configure applications.
## ------------------------------------------------------------------
##

import time
import os
import datetime
import logging
import threading
from functools import wraps
import platform
import psutil
import waptlicences
import certifi
import random
try:
    wapt_root_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
except:
    wapt_root_dir = 'c:/tranquilit/wapt'

#from waptutils import __version__

import sys
import locale
import json
import urllib.parse
import copy
import re

import configparser

# wapt specific stuff
from waptutils import ensure_unicode, update_ini_from_json_config
from waptutils import run
from waptutils import setloglevel
from waptutils import harakiri
from waptutils import datetime2isodate, jsondump, format_bytes

import common

import setuphelpers

from flask import request, Response, render_template, session

logger = logging.getLogger('waptservice')
tasks_logger = logging.getLogger('tasks')
setloglevel(tasks_logger,'info')

WAPTLOGGERS = ['waptcore', 'waptservice', 'waptws', 'waptdb', 'websocket', 'waitress', 'wapttasks', 'peercache', 'waptwua']

# i18n
import gettext
try:
    system_locale = locale.getdefaultlocale()[0].split('_',1)[0]
except:
    system_locale = 'en'
trans = gettext.translation(domain='messages', localedir=os.path.join(os.path.dirname(__file__),'translations'), languages=[system_locale,], fallback=True)
trans.install()
_ = trans.gettext

class WaptServiceRemoteAction(object):
    def __init__(self, name, action, required_attributes=[]):
        self.name = name
        self.action = action
        self.required_attributes = required_attributes

    def trigger_action(self, *args, **argv):
        self.action(*args, **argv)

waptservice_remote_actions = {}

_service_global_wapt = None

def scoped_wapt()->common.Wapt:
    """Flask request contextual cached Wapt instance access"""
    global _service_global_wapt
    if _service_global_wapt is None:
        _service_global_wapt = common.Wapt(config_filename=waptconfig.config_filename, wapt_base_dir=waptconfig.wapt_base_dir)
        if sys.platform == 'win32':
            apply_host_settings(waptconfig)
    # apply settings if changed at each wapt access...
    elif _service_global_wapt.reload_config_if_updated():
        # apply waptservice / waptexit specific settings
        if sys.platform == 'win32':
            apply_host_settings(waptconfig)
    #logger.debug('returning a scoped Wapt instance for flask')
    return _service_global_wapt

def apply_host_settings(waptconfig):
    """Apply waptservice / waptexit specific settings
    """
    wapt = common.Wapt(config_filename=waptconfig.config_filename, wapt_base_dir=waptconfig.wapt_base_dir)
    try:
        if waptconfig.max_gpo_script_wait is not None and wapt.max_gpo_script_wait != waptconfig.max_gpo_script_wait:
            logger.info('Setting max_gpo_script_wait to %s' % waptconfig.max_gpo_script_wait)
            wapt.max_gpo_script_wait = waptconfig.max_gpo_script_wait
        if waptconfig.pre_shutdown_timeout is not None and wapt.pre_shutdown_timeout != waptconfig.pre_shutdown_timeout:
            logger.info('Setting pre_shutdown_timeout to %s' % waptconfig.pre_shutdown_timeout)
            wapt.pre_shutdown_timeout = waptconfig.pre_shutdown_timeout
        if waptconfig.hiberboot_enabled is not None and wapt.hiberboot_enabled != waptconfig.hiberboot_enabled:
            logger.info('Setting hiberboot_enabled to %s' % waptconfig.hiberboot_enabled)
            wapt.hiberboot_enabled = waptconfig.hiberboot_enabled
    except Exception as e:
        logger.critical('Unable to set shutdown policies : %s' % e)


def register_remote_action(name, action, required_attributes=[]):
    waptservice_remote_actions[name] = WaptServiceRemoteAction(name, action, required_attributes)


def forbidden(msg=''):
    """Sends a 403 response that enables basic auth"""
    return Response(
        'Restricted access.\n%s\n' % msg,
        403)

def notfound():
    """Sends a 404"""
    return Response(
        'Not found.\n',
        404)

def badtarget():
    """Sends a 400 response if uuid mismatch"""
    return Response(
        'Host target UUID is not matching your request.\n',
        400)

def allow_local(f):
    """Restrict access to localhost"""
    @wraps(f)
    def decorated(*args, **kwargs):
        if not waptconfig.allow_html and (request.path.endswith('.html') or request.args.get('format') == 'html'):
            return forbidden()
        if request.remote_addr in ['127.0.0.1']:
            return f(*args, **kwargs)
        else:
            return forbidden()
    return decorated

def allow_local_token(f):
    """Restrict access to localhost authenticated with bearer only"""
    @wraps(f)
    def decorated(*args, **kwargs):
        if not waptconfig.allow_html and (request.path.endswith('.html') or request.args.get('format') == 'html'):
            return forbidden()

        if request.remote_addr in ['127.0.0.1']:
            if not request.authorization:
                return forbidden()

            bearer_token = request.authorization.token
            if not bearer_token:
                return forbidden()

            w = scoped_wapt()
            token_gen = w.get_secured_token_generator(SESSION_SECRET_KEY)
            try:
                max_age = w.token_lifetime
                token_content = token_gen.loads(bearer_token, max_age=max_age)
                session.username = token_content.get('username', [])
                session.usergroups = token_content.get('groups',[])
            except:
                # password is not a token or token is invalid
                return forbidden()
        else:
            return forbidden()
        return f(*args, **kwargs)
    return decorated

def authenticate(msg=None):
    """Sends a 401 response that enables basic auth"""
    if msg:
        return Response(msg, 401, {'WWW-Authenticate': 'Basic realm="Login Required"'})
    return Response(
        'Could not verify your access level for that URL.\n'
        'You have to login with proper credentials', 401,
        {'WWW-Authenticate': 'Basic realm="Login Required"'})

class WaptServiceConfig(object):
    """Configuration parameters from wapt-get.ini file
    >>> waptconfig = WaptServiceConfig('c:/wapt/wapt-get.ini')
    >>> waptconfig.load()
    """

    global_attributes = ['config_filename',
                         'MAX_HISTORY', 'waptservice_port',
                         'dbpath', 'loglevel', 'log_directory', 'waptserver',
                         'hiberboot_enabled', 'max_gpo_script_wait', 'pre_shutdown_timeout', 'log_to_windows_events',
                         'allow_user_service_restart', 'signature_clockskew', 'notify_user', 'waptservice_admin_filter',
                         'enable_remote_repo', 'local_repo_path', 'local_repo_sync_task_period', 'local_repo_time_for_sync_start',
                         'local_repo_time_for_sync_end', 'local_repo_limit_bandwidth', 'wol_port', 'wol_relay', 'service_auth_type',
                         'waptservice_poll_timeout','waptupdate_task_period',
                         'download_after_update_with_waptupdate_task_period','forced_installs_task_period',
                         'reconfig_on_network_change', 'websockets_verify_cert', 'update_server_status_on_connect', 'update_packages_on_connect', 'allow_html',
                         'disable_tls', 'waptwua_install_scheduling', 'waptwua_download_scheduling', 'waptwua_postboot_delay',"ad_domain_name"]
    list_config_attributes = common.Wapt.list_config_attributes

    MAX_HISTORY = 30
    allow_user_service_restart = False
    waptservice_port = 8088
    language = locale.getdefaultlocale()[0]
    loglevel = "warning"
    wapt_base_dir = wapt_root_dir
    log_directory = os.path.join(wapt_base_dir, 'log')
    log_to_windows_events = False
    private_dir = os.path.join(wapt_base_dir, 'private')
    public_dir = os.path.join(wapt_base_dir, 'public')
    packages_cache_dir = os.path.join(private_dir,'cache')
    wapt_temp_dir = None
    waptserver = None
    waptservice_poll_timeout = 20
    waptupdate_task_period = '120m'
    waptupgrade_task_period = None
    forced_installs_task_period = '2m'
    hiberboot_enabled = None
    max_gpo_script_wait = None
    pre_shutdown_timeout = None
    websockets_proto = None
    websockets_host = None
    websockets_port = None
    websockets_verify_cert = False
    websockets_low_level_reconnect = False
    websockets_ping = 10
    websockets_retry_delay = 60
    websockets_check_config_interval = 120
    websockets_hurry_interval = 1
    websockets_root = 'socket.io'
    websockets_request_timeout = 15000
    signature_clockskew = 6*60*60
    notify_user = False
    enable_remote_repo = False
    sync_only_forced = False
    enable_diff_repo = False
    local_repo_sync_task_period = '2h'
    remote_repo_dirs = ['wapt', 'waptwua','wads']
    local_repo_time_for_sync_start = None
    local_repo_time_for_sync_end = None
    local_repo_limit_bandwidth = None
    wol_port = '7,9'
    wol_relay = False
    service_auth_type = 'filetoken'
    waptservice_admin_filter = False # False = we assume local computer administrator as members of waptselfservice, so they see all packages
    waptservice_allow_all_packages = False # if True, all users can install or remove all packages (ie we assume the are member of waptselfservice)
    logon32_provider =  "LOGON32_PROVIDER_DEFAULT"
    ldap_auth_server = None
    ldap_auth_base_dn = None
    download_after_update_with_waptupdate_task_period = True
    log_retention_days = 30
    allow_remote_shutdown = False
    allow_remote_reboot = False
    minimum_battery_percent = 15
    reconfig_on_network_change = False
    update_server_status_on_connect = False # when websocket connect, enqueue a update_server_status task.
    update_packages_on_connect = False # when websocket connect, enqueue a update task.
    allow_html = False
    disable_tls = False # disable tls on local http service
    waptwua_download_scheduling = None
    waptwua_install_scheduling = None
    waptwua_postboot_delay = '10m'
    ad_domain_name = None
    peercache_enable = False

    def __init__(self, config_filename=None, wapt_base_dir=None):
        if wapt_base_dir is not None:
            self.wapt_base_dir = wapt_base_dir

        if not config_filename:
            self.config_filename = os.path.join(self.wapt_base_dir, 'wapt-get.ini')
        else:
            self.config_filename=config_filename

        self.private_dir = os.path.join(self.wapt_base_dir, 'private')
        self.public_dir = os.path.join(self.wapt_base_dir, 'public')

        self.dbpath = os.path.join(self.private_dir, 'db', 'waptdb.sqlite')
        for log in WAPTLOGGERS:
            setattr(self, 'loglevel_%s' % log, None)
            self.global_attributes.append('loglevel_%s' % log)

        self.log_directory = os.path.join(self.wapt_base_dir, 'log')
        if not os.path.exists(self.log_directory):
            os.mkdir(self.log_directory)


        self.waptserver = None
        self.configs_dir = None

        if self.config_filename:
            relative_configs_dir = os.path.join(os.path.dirname(self.config_filename),'conf.d')
            if os.path.isdir(relative_configs_dir):
                self.configs_dir = relative_configs_dir

        self.config_files_hash = None
        self.local_repo_path = os.path.join(self.wapt_base_dir, 'repository')

    def load(self, merge_config_packages=False, wapt_base_dir=None):
        if wapt_base_dir is not None:
            self.wapt_base_dir = wapt_base_dir
        """Load waptservice parameters from global wapt-get.ini file"""
        config = configparser.RawConfigParser()
        if not os.path.isfile(self.config_filename):
            # try to init an empty wapt-get.ini if we have the rights to.
            try:
                with open(self.config_filename,'w') as f:
                    f.write('[global]\n')
            except:
                raise Exception(_("FATAL. Couldn't initialize a config file : {}").format(self.config_filename))

        if os.path.isfile(self.config_filename):
            relative_configs_dir = os.path.join(os.path.dirname(self.config_filename),'conf.d')
            if os.path.isdir(relative_configs_dir):
                self.configs_dir = relative_configs_dir

            # merge config packages json file and rewrite config if previous config_hash differs from current config_hash
            if merge_config_packages and self.configs_dir:
                update_ini_from_json_config(self.config_filename, self.configs_dir, self.wapt_base_dir, merge_lists=self.list_config_attributes)
            config.read(self.config_filename)
        else:
            raise Exception(_("FATAL. Couldn't open config file : {}").format(self.config_filename))

        # lecture configuration
        if config.has_section('global'):
            if wapt_base_dir is None and config.has_option('global','wapt_base_dir') and config.get('global', 'wapt_base_dir') != '':
                self.wapt_base_dir = config.get('global','wapt_base_dir')

            if config.has_option('global', 'private_dir'):
                self.private_dir = config.get('global', 'private_dir')
            else:
                self.private_dir = os.path.join(self.wapt_base_dir,'private')

            if config.has_option('global', 'public_dir'):
                self.public_dir = config.get('global', 'public_dir')
            else:
                self.public_dir = os.path.join(self.wapt_base_dir,'public')

            if config.has_option('global', 'packages_cache_dir') and config.get('global', 'packages_cache_dir') != '':
                self.packages_cache_dir = config.get('global', 'packages_cache_dir')
            else:
                self.packages_cache_dir = os.path.join(self.private_dir, 'cache')

            if config.has_option('global', 'wapt_temp_dir'):
                self.wapt_temp_dir = config.get('global', 'wapt_temp_dir')

            if config.has_option('global', 'waptservice_admin_filter'):
                self.waptservice_admin_filter = config.getboolean('global', 'waptservice_admin_filter')
            else:
                self.waptservice_admin_filter = WaptServiceConfig.waptservice_admin_filter

            if config.has_option('global', 'waptservice_allow_all_packages'):
                self.waptservice_allow_all_packages = config.getboolean('global', 'waptservice_allow_all_packages')
            else:
                self.waptservice_allow_all_packages = WaptServiceConfig.waptservice_allow_all_packages

            if config.has_option('global', 'waptservice_port'):
                port = config.get('global', 'waptservice_port')
                if port:
                    self.waptservice_port = int(port)
                else:
                    self.waptservice_port = None
            else:
                self.waptservice_port = WaptServiceConfig.waptservice_port

            if config.has_option('global', 'language'):
                self.language = config.get('global', 'language')
            else:
                self.language = WaptServiceConfig.language

            if config.has_option('global', 'download_after_update_with_waptupdate_task_period'):
                self.download_after_update_with_waptupdate_task_period = config.getboolean('global', 'download_after_update_with_waptupdate_task_period')
            else:
                self.download_after_update_with_waptupdate_task_period = WaptServiceConfig.download_after_update_with_waptupdate_task_period

            if config.has_option('global', 'waptservice_poll_timeout'):
                self.waptservice_poll_timeout = int(config.get('global', 'waptservice_poll_timeout'))
            else:
                self.waptservice_poll_timeout = WaptServiceConfig.waptservice_poll_timeout

            if config.has_option('global', 'waptupgrade_task_period'):
                self.waptupgrade_task_period = config.get('global', 'waptupgrade_task_period') or None
            else:
                self.waptupgrade_task_period =  WaptServiceConfig.waptupgrade_task_period

            if config.has_option('global', 'waptupdate_task_period'):
                self.waptupdate_task_period = config.get('global', 'waptupdate_task_period') or None
            else:
                self.waptupdate_task_period = WaptServiceConfig.waptupdate_task_period

            if config.has_option('global', 'forced_installs_task_period'):
                self.forced_installs_task_period = config.get('global', 'forced_installs_task_period') or None
            else:
                self.forced_installs_task_period = WaptServiceConfig.forced_installs_task_period

            if config.has_option('global', 'dbpath'):
                self.dbpath = config.get('global', 'dbpath')
            else:
                self.dbpath = os.path.join(self.private_dir, 'db', 'waptdb.sqlite')
                # fallback for old wapt
                if not os.path.isfile(self.dbpath) and os.path.isfile(os.path.join(self.wapt_base_dir, 'db', 'waptdb.sqlite')):
                    self.dbpath = os.path.join(self.wapt_base_dir, 'db', 'waptdb.sqlite')

            if config.has_option('global', 'loglevel'):
                self.loglevel = config.get('global', 'loglevel')
            else:
                self.loglevel = WaptServiceConfig.loglevel

            for log in WAPTLOGGERS:
                if config.has_option('global', 'loglevel_%s' % log):
                    setattr(self, 'loglevel_%s' % log, config.get('global', 'loglevel_%s' % log))

            if config.has_option('global', 'log_to_windows_events'):
                self.log_to_windows_events = config.getboolean('global', 'log_to_windows_events')
            else:
                self.log_to_windows_events =  WaptServiceConfig.log_to_windows_events

            if config.has_option('global', 'allow_user_service_restart'):
                self.allow_user_service_restart = config.getboolean('global', 'allow_user_service_restart')
            else:
                self.allow_user_service_restart = WaptServiceConfig.allow_user_service_restart

            if config.has_option('global', 'notify_user'):
                self.notify_user = config.getboolean('global', 'notify_user')
            else:
                self.notify_user = WaptServiceConfig.notify_user

            if config.has_option('global', 'wol_port'):
                self.wol_port = config.get('global', 'wol_port')
            else:
                self.wol_port = WaptServiceConfig.wol_port

            if config.has_option('global', 'log_retention_days'):
                self.log_retention_days = config.get('global', 'log_retention_days')
            else:
                self.log_retention_days = WaptServiceConfig.log_retention_days

            if config.has_option('global', 'wapt_server'):
                self.waptserver = common.WaptServer().load_config(config)
                if self.waptserver.server_url:
                    waptserver_url = urllib.parse.urlparse(self.waptserver.server_url)
                    self.websockets_host = waptserver_url.hostname
                    self.websockets_proto = waptserver_url.scheme

                    if waptserver_url.port is None:
                        if waptserver_url.scheme == 'https':
                            self.websockets_port = 443
                        else:
                            self.websockets_port = 80
                    else:
                        self.websockets_port = waptserver_url.port

                    if waptserver_url.path in ('', '/'):
                        self.websockets_root = 'socket.io'
                    else:
                        self.websockets_root = '%s/socket.io' % waptserver_url.path[1:]

                    # TODO:  should be in WaptServer too.. ?
                    if config.has_option('global', 'websockets_verify_cert'):
                        self.websockets_verify_cert = config.get('global', 'websockets_verify_cert')
                        if self.websockets_verify_cert is None:
                            self.websockets_verify_cert = self.waptserver.verify_cert
                        if not isinstance(self.websockets_verify_cert,bool) and isinstance(self.websockets_verify_cert,str) and not self.websockets_verify_cert.lower() in ('1','0','true','false'):
                            if not os.path.isfile(self.websockets_verify_cert):
                                logger.warning('websockets_verify_cert certificate %s declared in configuration file can not be found. Waptserver websockets communication will fail' % self.websockets_verify_cert)
                        elif self.websockets_verify_cert in ('True','true','1',1):
                            self.websockets_verify_cert = True
                        elif self.websockets_verify_cert in ('False','false','0',0):
                            self.websockets_verify_cert = False
                    else:
                        self.websockets_verify_cert = self.waptserver.verify_cert

                    if isinstance(self.websockets_verify_cert,bool) and self.websockets_verify_cert==True:
                        self.websockets_verify_cert = certifi.where()

                else:
                    self.waptserver = None
                    self.websockets_host = None
                    self.websockets_proto = None
                    self.websockets_port = None
                    self.websockets_verify_cert = None
            else:
                self.waptserver = None
                self.websockets_host = None
                self.websockets_proto = None
                self.websockets_port = None
                self.websockets_verify_cert = None

            if config.has_option('global', 'websockets_ping'):
                self.websockets_ping = config.getint('global', 'websockets_ping')
            else:
                self.websockets_ping = WaptServiceConfig.websockets_ping

            if config.has_option('global', 'websockets_retry_delay'):
                self.websockets_retry_delay = config.getint('global', 'websockets_retry_delay')
            else:
                self.websockets_retry_delay = WaptServiceConfig.websockets_retry_delay

            if config.has_option('global', 'websockets_low_level_reconnect'):
                self.websockets_low_level_reconnect = config.getboolean('global','websockets_low_level_reconnect')
            else:
                self.websockets_low_level_reconnect = WaptServiceConfig.websockets_low_level_reconnect

            if config.has_option('global', 'websockets_check_config_interval'):
                self.websockets_check_config_interval = config.getint('global', 'websockets_check_config_interval')
            else:
                self.websockets_check_config_interval = WaptServiceConfig.websockets_check_config_interval

            if config.has_option('global', 'websockets_hurry_interval'):
                self.websockets_hurry_interval = config.getint('global', 'websockets_hurry_interval')
            else:
                self.websockets_hurry_interval = WaptServiceConfig.websockets_hurry_interval

            if config.has_option('global', 'signature_clockskew'):
                self.signature_clockskew = config.getint('global', 'signature_clockskew')
            else:
                self.signature_clockskew = WaptServiceConfig.signature_clockskew

            if config.has_option('repo-sync', 'enable_remote_repo'):
                self.enable_remote_repo = config.getboolean('repo-sync', 'enable_remote_repo')
                if self.enable_remote_repo:
                    if config.has_option('repo-sync', 'enable_diff_repo'):
                        self.enable_diff_repo = config.getboolean('repo-sync', 'enable_diff_repo')
                    if config.has_option('repo-sync', 'sync_only_forced'):
                        self.sync_only_forced = config.getboolean('repo-sync', 'sync_only_forced')
                    if config.has_option('repo-sync', 'remote_repo_dirs'):
                        self.remote_repo_dirs = config.get('repo-sync', 'remote_repo_dirs').replace(' ', '').split(',')
                    if config.has_option('repo-sync', 'local_repo_path'):
                        self.local_repo_path = config.get('repo-sync', 'local_repo_path')
                    if config.has_option('repo-sync', 'local_repo_time_for_sync_start'):
                        regex = re.compile('([0-1][0-9]|2[0-3]):[0-5][0-9]')
                        timeforsync_start = config.get('repo-sync', 'local_repo_time_for_sync_start')
                        if regex.match(timeforsync_start):
                            self.local_repo_time_for_sync_start = timeforsync_start
                            if config.has_option('repo-sync', 'local_repo_time_for_sync_end') and regex.match(config.get('repo-sync', 'local_repo_time_for_sync_end')):
                                self.local_repo_time_for_sync_end = config.get('repo-sync', 'local_repo_time_for_sync_end')
                            else:
                                self.local_repo_time_for_sync_end = '%s:%s' % (str((int(timeforsync_start.split(':')[0])+1)) if (int(timeforsync_start.split(':')[0])+1) > 9 else '0'+str((int(timeforsync_start.split(':')[0])+1)), timeforsync_start.split(':')[1])
                    elif config.has_option('repo-sync', 'local_repo_sync_task_period'):
                        self.local_repo_sync_task_period = config.get('repo-sync', 'local_repo_sync_task_period')
                    if config.has_option('repo-sync', 'local_repo_limit_bandwidth'):
                        self.local_repo_limit_bandwidth = config.getfloat('repo-sync', 'local_repo_limit_bandwidth')

            self.wol_relay = self.enable_remote_repo
            if config.has_option('global', 'wol_relay'):
                self.wol_relay = config.getboolean('global', 'wol_relay')
            # settings for waptexit / shutdown policy
            #   recommended settings :
            #       hiberboot_enabled = 0
            #       max_gpo_script_wait = 180
            #       pre_shutdown_timeout = 180
            for param in ('max_gpo_script_wait', 'pre_shutdown_timeout'):
                if config.has_option('global', param):
                    setattr(self, param, config.getint('global', param))
                else:
                    setattr(self, param, getattr(WaptServiceConfig,param))

            for param in ('hiberboot_enabled',):
                if config.has_option('global', param):
                    setattr(self, param, config.getboolean('global', param))
                else:
                    setattr(self, param, getattr(WaptServiceConfig,param))

            if config.has_option('global', 'service_auth_type'):
                self.service_auth_type = config.get('global', 'service_auth_type').lower()
            else:
                self.service_auth_type = WaptServiceConfig.service_auth_type

            if config.has_option('global', 'ldap_auth_server'):
                self.ldap_auth_server = config.get('global', 'ldap_auth_server')
            else:
                self.ldap_auth_server = WaptServiceConfig.ldap_auth_server

            if config.has_option('global', 'logon32_provider'):
                self.logon32_provider = config.get('global', 'logon32_provider')
            else:
                self.logon32_provider = WaptServiceConfig.logon32_provider

            if config.has_option('global', 'ldap_auth_base_dn'):
                self.ldap_auth_base_dn = config.get('global', 'ldap_auth_base_dn')
            else:
                self.ldap_auth_base_dn = WaptServiceConfig.ldap_auth_base_dn

            if config.has_option('global', 'ad_domain_name'):
                self.ad_domain_name = config.get('global', 'ad_domain_name')
            else:
                self.ad_domain_name = WaptServiceConfig.ad_domain_name

            if config.has_option('global', 'allow_remote_shutdown'):
                self.allow_remote_shutdown = config.getboolean('global', 'allow_remote_shutdown')
            else:
                self.allow_remote_shutdown = WaptServiceConfig.allow_remote_shutdown

            if config.has_option('global', 'allow_remote_reboot'):
                self.allow_remote_reboot = config.getboolean('global', 'allow_remote_reboot')
            else:
                self.allow_remote_reboot = WaptServiceConfig.allow_remote_reboot

            if config.has_option('global', 'minimum_battery_percent'):
                self.minimum_battery_percent = config.getint('global', 'minimum_battery_percent')
            else:
                self.minimum_battery_percent =  WaptServiceConfig.minimum_battery_percent

            if config.has_option('global', 'reconfig_on_network_change'):
                self.reconfig_on_network_change = config.getboolean('global', 'reconfig_on_network_change')
            else:
                self.reconfig_on_network_change = WaptServiceConfig.reconfig_on_network_change

            if config.has_option('global', 'update_server_status_on_connect'):
                self.update_server_status_on_connect = config.getboolean('global', 'update_server_status_on_connect')
            else:
                self.update_server_status_on_connect = WaptServiceConfig.update_server_status_on_connect

            if config.has_option('global', 'update_packages_on_connect'):
                self.update_packages_on_connect = config.getboolean('global', 'update_packages_on_connect')
            else:
                self.update_packages_on_connect = WaptServiceConfig.update_packages_on_connect

            if config.has_option('global', 'allow_html'):
                self.allow_html = config.getboolean('global', 'allow_html')
            else:
                self.allow_html = WaptServiceConfig.allow_html

            if config.has_option('global', 'peercache_enable'):
                self.peercache_enable = config.getboolean('global', 'peercache_enable')
            else:
                self.peercache_enable = WaptServiceConfig.peercache_enable

            if config.has_option('global', 'disable_tls'):
                self.disable_tls = config.getboolean('global', 'disable_tls')
            else:
                self.disable_tls = WaptServiceConfig.disable_tls

            if config.has_option('waptwua', 'download_scheduling'):
                self.waptwua_download_scheduling = config.get('waptwua', 'download_scheduling')
            else:
                self.waptwua_download_scheduling = WaptServiceConfig.waptwua_download_scheduling

            if config.has_option('waptwua', 'install_scheduling'):
                self.waptwua_install_scheduling = config.get('waptwua', 'install_scheduling')
            else:
                self.waptwua_install_scheduling = WaptServiceConfig.waptwua_install_scheduling

            if config.has_option('waptwua', 'postboot_delay'):
                self.waptwua_postboot_delay = config.get('waptwua', 'postboot_delay')
            else:
                self.waptwua_postboot_delay = WaptServiceConfig.waptwua_postboot_delay



        else:
            raise Exception(_("FATAL, configuration file {} has no section [global]. Please check Waptserver documentation").format(self.config_filename))

    def as_dict(self):
        result = {}
        for att in self.global_attributes:
            result[att] = getattr(self, att)
        return result

    def __str__(self):
        return "{}".format(self.as_dict(),)

# init translations
waptconfig = WaptServiceConfig()


class WaptEvent(object):
    """Store single event with list of subscribers"""
    DEFAULT_TTL = 20 * 60

    def __init__(self, event_type, data=None, owner=None):
        self.event_type = event_type
        self.data = copy.deepcopy(data)

        self.id = None
        self.ttl = self.DEFAULT_TTL
        self.date = time.time()
        # list of ids of subscribers which have not yet retrieved the event
        self.subscribers = []
        self.owner = owner

    def as_dict(self):
        return dict(
            id=self.id,
            event_type=self.event_type,
            date=self.date,
            data=self.data,
            owner=self.owner
        )


class WaptEvents(object):
    """Thread safe central list of last events so that consumer can get list
        of latest events using http long poll requests"""

    def __init__(self, max_history=300):
        self.max_history = max_history
        self.get_lock = threading.RLock()
        self.events = []
        self.subscribers = []

    def get_missed(self, last_read=None, max_count=None, owner=None):
        """returns events since last_read"""
        with self.get_lock:
            if last_read is None:
                return [e for e in self.events if (not e.owner or e.owner == owner)]
            else:
                if max_count is not None:
                    return [e for e in self.events[-max_count:] if e.id > last_read and (not e.owner or e.owner == owner)]  # pylint: disable=invalid-unary-operand-type
                else:
                    if not self.events or last_read > self.events[-1].id:
                        if not self.events:
                            last_read = 0
                        else:
                            last_read = self.events[-1].id
                    return [e for e in self.events if e.id > last_read and (not e.owner or e.owner == owner)]

    def last_event_id(self):
        with self.get_lock:
            if not self.events:
                return None
            else:
                return self.events[-1].id

    def put(self, item):
        with self.get_lock:
            try:
                if self.events:
                    last_event = self.events[-1]
                    if last_event == item:
                        return last_event
                else:
                    last_event = None

                if last_event:
                    item.id = last_event.id + 1
                else:
                    item.id = 0

                self.events.append(item)
                item.subscribers.extend(self.subscribers)
                # keep track of a global position for consumers
                if len(self.events) > self.max_history:
                    del self.events[:len(self.events) - self.max_history]
                return item
            finally:
                pass
                # self.event_available.notify_all()

    def post_event(self, event_type, data=None, owner=None):
        item = WaptEvent(event_type, data=data, owner=owner)
        return self.put(item)

    def cleanup(self):
        """Remove events with age>ttl"""
        with self.get_lock:
            for item in reversed(self.events):
                if item.date+item.ttl > time.time():
                    self.events.remove(item)


class EventsPrinter(object):
    '''EventsPrinter class which serves to emulates a file object and logs
       whatever it gets sent to a broadcast object at the INFO level.'''

    def __init__(self, events, logs):
        '''Grabs the specific brodcaster to use for printing.'''
        self.events = events
        self.logs = logs

    def write(self, text):
        '''Logs written output to listeners'''
        if text and text != '\n':
            if self.events:
                self.events.post_event('PRINT', ensure_unicode(text))
            self.logs.append(ensure_unicode(text))


def ensure_serializable(data):
    try:
        json.dumps(data)
        return data
    except:
        d = jsondump(data)
        return json.loads(d)

class WaptTask(object):
    """Base object class for all wapt task : download, install, remove, upgrade..."""

    def __init__(self, **args):
        self.id = -1
        self.task_manager = None
        self.priority = 100
        self.order = 0
        self.external_pids = []
        self.create_date = datetime.datetime.utcnow()
        self.start_not_before = None
        self.start_not_after = None
        self.start_date = None
        self.finish_date = None
        self.logs = []
        self.result = None
        self.summary = ""
        # from 0 to 100%
        self._progress = 0.0
        self._runstatus = ""
        self.notify_server_on_start = False
        self.notify_server_on_finish = False
        self.notify_user = False
        self.created_by = None
        self.force = False

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

        self.lang = None

        self._last_status_time = 0.0

    @property
    def wapt(self) -> common.Wapt:
        return self.task_manager and self.task_manager.wapt or None

    @property
    def progress(self):
        return self._progress

    @progress.setter
    def progress(self, value):
        self._progress = value
        if time.time() - self._last_status_time >= 1:
            if self.task_manager.events:
                self.task_manager.events.post_event('TASK_STATUS', self.as_dict(), owner=self.created_by)
                self._last_status_time = time.time()
        if self.task_manager._wapt.task_is_cancelled.is_set():
            raise Exception('Task cancelled')

    @property
    def runstatus(self):
        return self._runstatus

    @runstatus.setter
    def runstatus(self, value):
        logger.debug(value)
        self._runstatus = value
        if self.wapt:
            self.wapt.runstatus = value
            if time.time() - self._last_status_time >= 1.0:
                if self.task_manager.events:
                    self.task_manager.events.post_event('TASK_STATUS', self.as_dict(), owner=self.created_by)
                    self._last_status_time = time.time()

    def update_status(self, status):
        """Update runstatus in database and send PROGRESS event"""
        self.runstatus = status

    def can_run(self, explain=False):
        """Return True if all the requirements for the task are met
        (ex. install can start if package+depencies are downloaded)"""
        return True

    def _run(self):
        """method to override in descendant to do the actual work"""
        pass

    def run(self):
        """register start and finish time, call _run, redirect stdout and stderr to events broadcaster
            result of task should be stored in self.result
            human readable summary of work done should be stored in self.summary
        """
        self.start_date = datetime.datetime.utcnow()
        try:
            if self.wapt:
                self.wapt.task_is_cancelled.clear()
                # to keep track of external processes launched by Wapt.run()
                self.wapt.pidlist = self.external_pids
            self._run()
            self._progress = 100.0
        finally:
            self.finish_date = datetime.datetime.utcnow()
            if self.wapt and self.task_manager.events:
                self.task_manager.events.post_event('TASK_STATUS', self.as_dict(), owner=self.created_by)

    def kill(self):
        """if task has been started, kill the task (ex: kill the external processes"""
        self.summary = _('Canceled')
        self.logs.append('Canceled')

        if self.task_manager and self.task_manager._wapt:  # we access directly to avoid checking for thread ownership
            self.task_manager._wapt.task_is_cancelled.set()
        if self.external_pids:
            for pid in self.external_pids:
                logger.debug('Killing process with pid {}'.format(pid))
                setuphelpers.killtree(pid)
            del(self.external_pids[:])
        raise Exception(_('Task canceled %s') % self.id)

    def run_external(self, *args, **kwargs):
        """Run an external process, register pid in current task to be able to kill it"""
        return run(*args, pidlist=self.external_pids, **kwargs)

    def __str__(self):
        return _("{classname} {id} created {create_date} started:{start_date} finished:{finish_date} ").format(**self.as_dict())

    def as_dict(self):
        return copy.deepcopy(dict(
            id=self.id,
            classname=self.__class__.__name__,
            priority=self.priority,
            order=self.order,
            force=self.force,
            create_date=self.create_date and self.create_date.isoformat(),
            start_date=self.start_date and self.start_date.isoformat(),
            start_not_before=self.start_not_before,
            start_not_after=self.start_not_after,
            finish_date=self.finish_date and self.finish_date.isoformat(),
            logs='\n'.join([ensure_unicode(l) for l in self.logs]),
            result=ensure_serializable(self.result),
            summary=self.summary,
            progress=self.progress,
            runstatus=self.runstatus,
            description="{}".format(self),
            pidlist="{0}".format(self.external_pids),
            notify_user=self.notify_user,
            notify_server_on_start=self.notify_server_on_start,
            notify_server_on_finish=self.notify_server_on_finish,
            created_by=self.created_by,
        ))

    def as_json(self):
        return json.dumps(self.as_dict(), indent=True)

    def __repr__(self):
        return "<{}>".format(self)

    def __cmp__(self, other):

        def cmp(a, b):
            return (a > b) - (a < b)

        now_utc = datetime2isodate()
        if self.start_not_before and self.start_not_before < now_utc:
            nb1 = now_utc
        else:
            nb1 = self.start_not_before or now_utc

        if other.start_not_before and other.start_not_before < now_utc:
            nb2 = now_utc
        else:
            nb2 = other.start_not_before or now_utc

        if self.start_not_after and self.start_not_after > now_utc:
            na1 = now_utc
        else:
            na1 = self.start_not_after or now_utc

        if other.start_not_after and other.start_not_after > now_utc:
            na2 = now_utc
        else:
            na2 = other.start_not_after or now_utc

        return cmp((self.priority, nb1, na1, self.order), (other.priority, nb2, na2, other.order))

    def __eq__(self, other):
        return self.__cmp__(other) == 0

    def __ne__(self, other):
        return self.__cmp__(other) != 0

    def __gt__(self, other):
        return self.__cmp__(other) > 0

    def __ge__(self, other):
        return self.__cmp__(other) >= 0

    def __lt__(self, other):
        return self.__cmp__(other) < 0

    def __le__(self, other):
        return self.__cmp__(other) <= 0

    def same_action(self, other):
        return self.__class__ == other.__class__

class WaptNetworkReconfig(WaptTask):
    def __init__(self, **args):
        super(WaptNetworkReconfig, self).__init__()
        self.priority = 0
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        logger.warning('Reloading config file')
        self.status = _('Reloading config file')
        self.wapt.load_config(waptconfig.config_filename)
        self.wapt.network_reconfigure()
        waptconfig.load(True)
        self.update_status(_('Config file reloaded'))
        self.result = waptconfig.as_dict()
        self.notify_server_on_finish = self.wapt.waptserver_available()

    def __str__(self):
        return _("Reconfiguring network access")

    def same_action(self, other):
        return (self.__class__ == other.__class__)

class WaptClientUpgrade(WaptTask):
    def __init__(self, **args):
        super(WaptClientUpgrade, self).__init__()
        self.priority = 10
        self.notify_server_on_start = True
        self.notify_server_on_finish = False
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        """Launch an external 'wapt-get waptupgrade' process to upgrade local copy of wapt client"""
        from setuphelpers import run
        output = ensure_unicode(run('"%s" %s' % (os.path.join(wapt_root_dir, 'wapt-get.exe'), 'waptupgrade')))
        self.result = {'result': 'OK', 'message': output}

    def __str__(self):
        return _("Upgrading WAPT client")


class WaptServiceRestart(WaptTask):
    """A task to restart the waptservice using a spawned cmd process"""

    def __init__(self, **args):
        super(WaptServiceRestart, self).__init__()
        self.priority = 10000

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        """Launch an external 'wapt-get waptupgrade' process to upgrade local copy of wapt client"""
        result = 'ERROR'
        if platform.system() == 'Windows':
            result = 'OK'
            harakiri(10)
        elif platform.system() == 'Darwin':
            output = setuphelpers.run("launchctl kickstart -k system/it.tranquil.waptservice")
            result = 'OK'
        elif platform.system() == 'Linux':
            output = setuphelpers.run('systemctl restart waptservice')
            result = 'OK'
        else:
            output = 'Restart not supported'
            result = 'ERROR'
        logger.info(output)
        self.result = {'result': result, 'message': output}

    def __str__(self):
        return _("Restarting local WAPT service")


class WaptUpdate(WaptTask):
    def __init__(self, **args):
        super(WaptUpdate, self).__init__()
        self.priority = 10
        self.notify_server_on_finish = True
        self.force = False
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        self.wapt.check_install_running()
        self.progress = 0
        print(_('Get packages index from %s') % self.wapt.repositories)
        self.result = self.wapt.update(force=self.force, register=self.notify_server_on_finish)
        """result: {
            count: 176,
            added: [ ],
            repos: [
            "http://srvwapt.tranquilit.local/wapt",
            "http://srvwapt.tranquilit.local/wapt-host"
            ],
            upgrades: ['install': 'additional': 'upgrade': ],
            date: "2014-02-28T19:30:35.829000",
            removed: [ ]
        },"""
        s = []
        if len(self.result['added']) > 0:
            s.append(_('{} new package(s)').format(len(self.result['added'])))
        if len(self.result['removed']) > 0:
            s.append(_('{} removed package(s)').format(len(self.result['removed'])))
        s.append(_('{} package(s) in the repository').format(self.result['count']))
        all_install = self.result['upgrades']['install'] +\
            self.result['upgrades']['additional'] +\
            self.result['upgrades']['upgrade']
        installs = ','.join(all_install)
        removes = ','.join(self.result['upgrades']['remove'])
        errors = ','.join([p.asrequirement() for p in self.wapt.error_packages()])
        if installs:
            s.append(_('Packages to be updated : {}').format(installs))
        if removes:
            s.append(_('Packages to be removed : {}').format(removes))
        if errors:
            s.append(_('Packages with errors : {}').format(errors))
        if not installs and not errors and not removes:
            s.append(_('System up-to-date'))
        logger.debug(('\n'.join(s)))
        self.summary = '\n'.join(s)

    def __str__(self):
        return _("Updating available packages")


class WaptDownloadUpgrade(WaptTask):
    def __init__(self, **args):
        super(WaptDownloadUpgrade, self).__init__()
        self.force = False
        self.from_websocket = False

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        if not self.force:
            check_battery()
        def cjoin(l):
            return ','.join(['%s' % ensure_unicode(p) for p in l])

        all_tasks = []
        reqs = self.wapt.check_downloads()
        self.result = reqs

        if reqs and self.from_websocket :
            if waptconfig.peercache_enable :
                time.sleep(random.uniform(0, 2))

        for req in reqs:
            all_tasks.append(
                self.task_manager.add_task(
                    WaptDownloadPackage(
                        req.asrequirement(), force=self.force, notify_user=self.notify_user, created_by = self.created_by)
                        ).as_dict())

        self.summary = "Enqueued %s download tasks" % len(reqs)

    def __str__(self):
        return _('Download Upgradable packages')


class WaptUpgrade(WaptTask):
    def __init__(self, only_priorities=None, only_if_not_process_running=False, **args):
        super(WaptUpgrade, self).__init__()
        self.only_priorities = only_priorities
        self.only_if_not_process_running = only_if_not_process_running
        self.force = False

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

        # for subtasks
        self.subtasks_notify_server_on_finish = self.notify_server_on_finish
        self.notify_server_on_finish = False # we will insert such a task anyway

    def _run(self):
        if not self.force:
            check_battery()
        def cjoin(l):
            return ','.join(['%s' % ensure_unicode(p) for p in l])

        all_tasks = []
        actions = self.wapt.list_upgrade()
        self.result = actions
        to_install = actions['upgrade']+actions['additional']+actions['install']
        to_remove = actions['remove']

        for req in to_remove:
            all_tasks.append(self.task_manager.add_task(WaptPackageRemove(req, created_by=self.created_by, force=self.force, notify_user=self.notify_user,
                                                                          only_priorities=self.only_priorities, notify_server_on_finish=self.subtasks_notify_server_on_finish,
                                                                          only_if_not_process_running=self.only_if_not_process_running)).as_dict())

        for req in to_install:
            all_tasks.append(self.task_manager.add_task(WaptPackageInstall(req, created_by=self.created_by, force=self.force, notify_user=self.notify_user,
                                                                           only_priorities=self.only_priorities, notify_server_on_finish=self.subtasks_notify_server_on_finish,
                                                                           only_if_not_process_running=self.only_if_not_process_running,
                                                                           # we don't reprocess depends
                                                                           process_dependencies=True)).as_dict())

        if to_install and self.wapt.is_enterprise():
            all_tasks.append(self.task_manager.add_task(
                WaptAuditPackage(to_install,
                                 notify_server_on_finish=self.subtasks_notify_server_on_finish,
                                 created_by=self.created_by,
                                 force=self.force,
                                 notify_user=self.notify_user)).as_dict())

        all_tasks.append(self.task_manager.add_task(
                WaptUpdateServerStatus(
                                created_by=self.created_by,
                                notify_user=self.notify_user)).as_dict())

        all_install = self.result.get('install', [])
        if self.result.get('additional', []):
            all_install.extend(self.result['additional'])
        install = cjoin(all_install)
        upgrade = cjoin(self.result.get('upgrade', []))
        unavailable = ','.join([p[0] for p in self.result.get('unavailable', [])])
        s = []
        if install:
            s.append(_('Installed : {}').format(install))
        if upgrade:
            s.append(_('To upgrade : {}').format(upgrade))
        if unavailable:
            s.append(_('Unavailable : {}').format(unavailable))
        if not unavailable and not install and not upgrade:
            s.append(_('System up-to-date'))
        self.summary = "\n".join(s)

    def __str__(self):
        return _('Upgrade the packages installed on host')


class WaptUpdateServerStatus(WaptTask):
    """Send workstation status to server"""

    def __init__(self, **args):
        super(WaptUpdateServerStatus, self).__init__()

        self.force = False
        self.excluded_keys = []

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        errors = []
        self.result = self.wapt.update_server_status(force=self.force, excluded_keys=self.excluded_keys, errors = errors)
        if self.result is None:
            raise Exception(_('Unable to build or send current status to WAPT Server %s. Check logs. Errors: %s') % (self.wapt.waptserver.server_url, '\n'.join(errors)))
        self.summary = _('WAPT Server %s has been notified') % self.wapt.waptserver.server_url
        print('Done.')

    def __str__(self):
        return _("Update server with this host's status")


class WaptRegisterComputer(WaptTask):
    """Send workstation status to server"""

    def __init__(self, computer_description=None, **args):
        super(WaptRegisterComputer, self).__init__(**args)

        self.computer_description = computer_description

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        if self.wapt.waptserver_available():
            self.update_status(_('Sending computer status to waptserver'))
            try:
                self.result = self.wapt.register_computer(description=self.computer_description)
                self.progress = 50
                self.summary = _("Inventory has been sent to the WAPT server")
            except Exception as e:
                self.result = {}
                self.summary = _("Error while sending inventory to the server : {}").format(ensure_unicode(e))
                raise
        else:
            self.result = {}
            self.summary = _('WAPT Server is not available')
            raise Exception(self.summary)

    def __str__(self):
        return _("Update server with this host's inventory")

class WaptUnregisterComputer(WaptTask):
    """Unregister workstation from the server"""

    def __init__(self, computer_description=None, **args):
        super(WaptUnregisterComputer, self).__init__(**args)
        self.priority = 0

        self.computer_description = computer_description
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        try:
            self.result = self.wapt.unregister_computer()
            self.progress = 50
            self.summary = _("Computer has been unregistered from WAPT server")
        except Exception as e:
            self.result = {}
            self.summary = _("Error while unregistering host from the server : {}").format(ensure_unicode(e))
            raise

    def __str__(self):
        return _("Unregister host from wapt server")

class WaptCleanup(WaptTask):
    """Cleanup local packages cache"""

    def __init__(self, **args):
        super(WaptCleanup, self).__init__()
        self.priority = 1000

        self.force = False
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        def cjoin(l):
            return ','.join(['%s' % p for p in l])
        try:
            self.result = self.wapt.cleanup(obsolete_only=not self.force)
            self.summary = _("Packages erased : {}").format(cjoin(self.result))
        except Exception as e:
            self.result = {}
            self.summary = _("Error while clearing local cache : {}").format(ensure_unicode(e))
            raise Exception(self.summary)

    def __str__(self):
        return _("Clear local package cache")


class WaptLongTask(WaptTask):
    """Test action for debug purpose"""

    def __init__(self, **args):
        super(WaptLongTask, self).__init__()
        self.duration = 60
        self.raise_error = False

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        self.progress = 0.0
        for i in range(self.duration):
            if self.wapt:
                self.wapt.check_cancelled()
            # print u"Step {}".format(i)
            self.update_status("Step {}".format(i))
            self.progress = 100.0 / self.duration * i
            #print(("test {:.0f}%".format(self.progress)))
            time.sleep(1)
        if self.raise_error:
            raise Exception(_('raising an error for Test WaptLongTask'))

    def same_action(self, other):
        return (self.__class__ == other.__class__) and (self.duration == other.duration)

    def __str__(self):
        return _("Test long running task of {}s").format(self.duration)


class WaptDownloadPackage(WaptTask):
    def __init__(self, packagenames, usecache=True, **args):
        super(WaptDownloadPackage, self).__init__()
        if not isinstance(packagenames, list):
            self.packagenames = [packagenames]
        else:
            self.packagenames = packagenames
        self.usecache = usecache
        self.size = 0
        self._last_url = ''
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def printhook(self, received, total, speed, url):
        self.wapt.check_cancelled()
        if total > 1.0:
            stat = '%i / %i (%.0f%%) (%s/s)\r' % (received, total, 100.0*received/total, format_bytes(speed))
            self.progress = 100.0*received/total
            if not self.size:
                self.size = total
        else:
            stat = ''
        self._last_url = url
        self.update_status(_('Downloading %s : %s') % (url, stat))

    def _run(self):
        if not self.force:
            check_battery()
        self.update_status(_('Downloading %s') % (','.join(self.packagenames)))
        start = time.time()
        self.result = self.wapt.download_packages(self.packagenames, usecache=self.usecache, printhook=self.printhook)
        end = time.time()
        if self.result['errors']:
            self.summary = _("Error while downloading {packagenames}: {error}").format(packagenames=','.join(self.packagenames), error=self.result['errors'][0][1])
        else:
            if end-start > 0.01:
                self.summary = _("Done downloading {packagenames}. {speed}/s {last_url}").format(packagenames=','.join(self.packagenames), speed=format_bytes(self.size/(end-start)), last_url=self._last_url)
            else:
                self.summary = _("Done downloading {packagenames} {last_url}.").format(packagenames=','.join(self.packagenames), last_url=self._last_url)

    def as_dict(self):
        d = WaptTask.as_dict(self)
        d.update(
            dict(
                packagenames=self.packagenames,
                usecache=self.usecache,
            )
        )
        return d

    def __str__(self):
        return _("Download of {packagenames} (task #{id})").format(classname=self.__class__.__name__, id=self.id, packagenames=','.join(self.packagenames))

    def same_action(self, other):
        return (self.__class__ == other.__class__) and (self.packagenames == other.packagenames)




class WaptPackageInstall(WaptTask):

    def __init__(self, packagenames=None, force=False, only_priorities=None, only_if_not_process_running=False, process_dependencies=True, **args):
        super(WaptPackageInstall, self).__init__()
        self.packagenames = None

        if packagenames is not None:
            if not isinstance(packagenames, list):
                self.packagenames = [packagenames]
            else:
                self.packagenames = packagenames
        else:
            raise Exception('Missing packagenames parameter')
        self.force = force
        self.only_priorities = only_priorities
        self.only_if_not_process_running = only_if_not_process_running
        self.process_dependencies = process_dependencies

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        if not self.force:
            check_battery()
        self.update_status(_('Installing %s') % (','.join(self.packagenames)))

        def cjoin(l):
            return ','.join(["%s" % (p[1].asrequirement() if p[1] else p[0],) for p in l])

        self.result = self.wapt.install(apackages =self.packagenames,
                                        force=self.force,
                                        installed_by=self.created_by,
                                        only_priorities=self.only_priorities,
                                        only_if_not_process_running=self.only_if_not_process_running,
                                        process_dependencies=self.process_dependencies,
                                        raise_exception_if_errors_or_unavailable=False)

        all_install = self.result.get('install', [])
        if self.result.get('additional', []):
            all_install.extend(self.result['additional'])
        install = cjoin(all_install)
        upgrade = cjoin(self.result.get('upgrade', []))
        #skipped = cjoin(self.result['skipped'])
        errors = cjoin(self.result.get('errors', []))
        unavailable = cjoin(self.result.get('unavailable', []))
        s = []
        if install:
            s.append(_('Installed : {}').format(install))
        if upgrade:
            s.append(_('Updated : {}').format(upgrade))
        if errors:
            s.append(_('Errors : {}').format(errors))
        if unavailable:
            s.append(_('Unavailable : {}').format(unavailable))
        self.summary = "\n".join(s)
        if self.result.get('errors', []):
            raise Exception(_('Error during install of {}: errors in packages {}').format(
                self.packagenames,
                self.result.get('errors', [])))

    def __str__(self):
        return _("Installation of {packagenames} (task #{id})").format(classname=self.__class__.__name__, id=self.id, packagenames=','.join(self.packagenames))

    def same_action(self, other):
        return (self.__class__ == other.__class__) and (self.packagenames == other.packagenames)


class WaptPackageRemove(WaptPackageInstall):

    def _run(self):
        if not self.force:
            check_battery()
        def cjoin(l):
            return ','.join(['%s' % ensure_unicode(p) for p in l])

        self.result = self.wapt.remove(self.packagenames,
                                       force=self.force,
                                       only_if_not_process_running=self.only_if_not_process_running,
                                       only_priorities=self.only_priorities,uninstalled_by=self.created_by)

        s = []
        if self.result['removed']:
            s.append(_('Removed : {}').format(cjoin(self.result['removed'])))
        if self.result['errors']:
            s.append(_('Errors : {}').format(cjoin(self.result['errors'])))
        self.summary = "\n".join(s)

    def __str__(self):
        return _("Uninstall of {packagenames} (task #{id})").format(classname=self.__class__.__name__, id=self.id, packagenames=','.join(self.packagenames))


class WaptPackageForget(WaptTask):
    def __init__(self, packagenames, **args):
        super(WaptPackageForget, self).__init__()
        if not isinstance(packagenames, list):
            self.packagenames = [packagenames]
        else:
            self.packagenames = packagenames

        self.force = False

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        self.update_status(_('Forgetting %s') % self.packagenames)
        self.result = self.wapt.forget_packages(self.packagenames)
        if self.result:
            self.summary = _("Packages removed from database : %s") % ("\n".join(self.result),)
        else:
            self.summary = _("No package removed from database.")

    def __str__(self):
        return _("Forget {packagenames} (task #{id})").format(classname=self.__class__.__name__, id=self.id, packagenames=','.join(self.packagenames))

    def same_action(self, other):
        return (self.__class__ == other.__class__) and (self.packagenames == other.packagenames)


class WaptAuditPackage(WaptTask):
    def __init__(self, packagenames, **args):
        self.ignore_schedule = False
        super(WaptAuditPackage, self).__init__()
        if not isinstance(packagenames, list):
            self.packagenames = [packagenames]
        else:
            self.packagenames = packagenames

        self.force = False

        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        self.result = []
        self.progress = 0.0
        if self.packagenames:
            astep = 100.0 / len(self.packagenames)
            for package in self.packagenames:
                self.update_status(_('Auditing %s') % package)
                try:
                    self.result.append('%s: %s' % (package, self.wapt.audit(package, force=self.force, ignore_schedule=self.ignore_schedule, audited_by=self.created_by)))
                except Exception as e:
                    self.result.append('%s: error %s' % (package, e))
                self.progress += astep

        self.progress = 100.0
        self.update_status(_('Audit finished'))
        if self.result:
            self.summary = _("Audit result : %s") % ('\n'.join(self.result))
        else:
            self.summary = _("Audit processed (no output) for %s") % (self.packagenames,)

    def __str__(self):
        if len(self.packagenames) > 3:
            desc = _('%s packages') % len(self.packagenames)
        else:
            desc = ','.join(self.packagenames)
        return _("Audit of {packagenames} (task #{id})").format(classname=self.__class__.__name__, id=self.id, packagenames=desc)

    def same_action(self, other):
        return (self.__class__ == other.__class__) and (self.packagenames == other.packagenames)


class WaptDownloadIcon(WaptTask):
    def __init__(self, target_packages, usecache=True, **args):
        super(WaptDownloadIcon, self).__init__()

        if not isinstance(target_packages, list):
            self.target_packages = [target_packages]
        else:
            self.target_packages = target_packages # list of {'package','icon_sha256sum'}
        self.usecache = usecache
        self.size = 0
        self.last_downloaded = ''
        for k in args:
            if hasattr(self,k):
                setattr(self, k, args[k])
            else:
                logger.critical('Unknown %s argument %s' % (self.__class__,k))

    def _run(self):
        self.update_status(_('Downloading %s icons') % (len(self.target_packages),))

        start = time.time()

        target_dir = os.path.join(waptconfig.wapt_base_dir,'cache','icons')
        group_by_repo = {}
        for p in self.target_packages:
            if p['repo'] in group_by_repo:
                group_by_repo[p['repo']].append(p)
            else:
                group_by_repo[p['repo']] = [p]

        for repo,packages in group_by_repo.items():
            self.result = self.wapt.get_repo(repo).download_icons(packages,target_dir=target_dir)

        end = time.time()
        if end-start > 0.01:
            self.summary = _("Done downloading {count} icons. {speed}/s").format(count=len(self.target_packages), speed=format_bytes(self.size/(end-start)))
        else:
            self.summary = _("Done downloading icons {count}.").format(count=len(self.target_packages))

    def __str__(self):
        return _("Installation of icons for {count} packages (task #{id})").format(classname=self.__class__.__name__, id=self.id, count=len(self.target_packages))

    def as_dict(self):
        d = WaptTask.as_dict(self)
        d.update(
            nb_packages=len(self.target_packages),
            usecache=self.usecache,
            last_downloaded=self.last_downloaded
        )
        return d



def render_wapt_template(template_name_or_list, **context):
    global _
    global gettext

    if not '_' in context:
        context['_'] = _
    if not 'gettext' in context:
        context['gettext'] = gettext
    return render_template(template_name_or_list, **context)

def check_battery():
    """Generates an exception if the computer is not on power and the minimum
    battery level is not reached. The minimum_battery_percent parameter
    of the wapt-get.ini file controls this value."""
    if hasattr(psutil,"sensors_battery"):
        try:
            battery = psutil.sensors_battery()
            if hasattr(battery,"power_plugged"):
                if not battery.power_plugged :
                    if battery.percent < waptconfig.minimum_battery_percent:
                        raise Exception(_('Battery level too low %s, minimum battery %s' % (battery.percent,waptconfig.minimum_battery_percent)))
        except:
            return


SESSION_SECRET_KEY = waptlicences.random_password(64)
