# -*- coding: utf-8 -*-
##
## -----------------------------------------------------------------
##    This file is part of WAPT Software Deployment
##    Copyright (C) 2012 - 2024  Tranquil IT https://www.tranquil.it
##    All Rights Reserved.
##
##    WAPT helps systems administrators to efficiently deploy
##    setup, update and configure applications.
## ------------------------------------------------------------------
##

from typing import List,Dict
import os
import sys

if "__file__" in locals():
    sys.path.insert(0, os.path.realpath(os.path.join( os.path.dirname(os.path.realpath(__file__)),'..')))
# there is a security bug in this module. known files are by default ['/etc/mime.types', '/etc/... even on winddows
# this allow a regular user to trigger memory overflow if he create a huge file /etc/mime.type
import mimetypes
mimetypes.knownfiles.clear()

from waptservice.waptservice_common import WaptTask, WaptUpgrade, WaptUpdate, WaptUpdateServerStatus, WaptCleanup, WaptDownloadPackage, WaptLongTask
from waptservice.waptservice_common import WaptAuditPackage, WaptPackageRemove, WaptDownloadIcon, WaptDownloadUpgrade

from waptservice.waptservice_common import WaptServiceConfig, WaptServiceRestart, WaptNetworkReconfig, WaptPackageInstall
from waptservice.waptservice_common import forbidden, authenticate, allow_local, render_wapt_template
from waptservice.waptservice_common import WaptEvents, WaptRegisterComputer, WaptPackageForget
from waptservice.waptservice_common import waptconfig, WAPTLOGGERS, scoped_wapt, allow_local_token, apply_host_settings, SESSION_SECRET_KEY
from waptservice.waptservice_common import _

from waptpackage import PackageEntry
from waptcrypto import SSLPrivateKey
from flask import request, Flask, Response, send_from_directory, session, g, redirect
from waptservice.waptservice_socketio import WaptSocketIOClient
from setuphelpers import Version

if sys.platform == 'win32':
    from setuphelpers import get_computer_domain
else:
    from setuphelpers_unix import get_domain_hostname_from_keytab

import setuphelpers
from common import Wapt
import common
from waptutils import setloglevel, ensure_list, ensure_unicode, jsondump, LogOutput, get_time_delta, is_between_two_times, run, makepath
from waptutils import harakiri, user_home_directory, user_is_local_admin, user_is_member_of
from waptutils import EWaptAuthException
from waptutils import datetime2isodate, sanitize_filename, generate_unique_string, write_user_protected_file
from waptutils import SAFE_CIPHERS
from waptutils import in_case_insensitive_list

import ctypes
import datetime
import copy
import traceback
import queue
import threading
import json
import gc
import sqlite3
import logging
import cheroot.wsgi
from cheroot.ssl.builtin import BuiltinSSLAdapter
import ssl
from optparse import OptionParser
from waptutils import __version__, WAPT_VERSION_FULL
import time
import base64
import glob
import auth_module_ad
import waptlicences
import pyldap
import psutil

previous_garbage_count = {}
garbage_count = {}

v = (sys.version_info.major, sys.version_info.minor)
if v[0] != 3:
    raise Exception('wapt-get supports only Python 3, not %d.%d' % v)

try:
    wapt_root_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
except NameError:
    wapt_root_dir = 'c:/tranquilit/wapt'

if sys.platform.startswith('linux') or sys.platform.startswith('darwin'):
    if 'PYTHONPATH' in os.environ:
        del os.environ['PYTHONPATH']
    if 'PYTHONHOME' in os.environ:
        del os.environ['PYTHONHOME']

if not sys.platform == 'win32':
    import setproctitle
    setproctitle.setproctitle('waptservice')

if sys.platform == 'win32':
    import win32api
    import pythoncom
    import win32security
else:
    import pam, pwd
    from setuphelpers_unix import get_groups

from setuphelpers import local_users

if sys.platform == 'win32':
    from waptservice.enterprise import WaptWUAScanTask, WaptWUADowloadTask, WaptWUAInstallTask, waptwua_api
    import pywintypes
    import pywaptwua
else:
    pywaptwua=None

from waptservice.enterprise import WaptRunSessionSetup
from waptservice.repositories import WaptSyncRepo

logger = logging.getLogger('waptservice')
tasks_logger = logging.getLogger('wapttasks')

app = Flask(__name__)
app.config['PROPAGATE_EXCEPTIONS'] = True
# ephemeral cookies key
app.config['SECRET_KEY'] = SESSION_SECRET_KEY
app.task_manager: "WaptTaskManager" = None

app.name = 'waptservice'

if sys.platform == 'win32':
    app.register_blueprint(waptwua_api)

app.waptconfig: WaptServiceConfig = waptconfig

@app.teardown_appcontext
def close_connection(exception):
    try:
        local_wapt = getattr(g, 'wapt', None)
        if local_wapt is not None and local_wapt._waptdb and local_wapt._waptdb.transaction_depth > 0:
            try:
                local_wapt._waptdb.commit()
                local_wapt._waptdb = None
            except:
                try:
                    local_wapt._waptdb.rollback()
                    local_wapt._waptdb = None
                except:
                    local_wapt._waptdb = None

    except Exception as e:
        logger.debug('Error in teardown, please consider upgrading Flask if <0.10. %s' % e)


def check_auth_groups(wapt: Wapt, auth_header:str):
    """Authenticate a user and returns the self-service groups membership

    Args:
        auth_header (str): encoded Authorization header
        check_for_groups (list): restrict the search to this list of self service groups

    Returns:
        list: (username,list of user's self service groups memberships ex: ['compta','tech'])
    """

    # Domain admins and local admins are allowed
    # make a copy to not change global settings
    try:
        # try to get auth and groups from Bearer token
        if wapt is None:
            wapt = scoped_wapt()
        auths = [b.strip() for b in auth_header.split(',')]
        for auth in auths:
            (bearer,token) = auth.split(' ',1)
            if bearer == 'Bearer':
                token_gen = wapt.get_secured_token_generator(app.config['SECRET_KEY'])
                try:
                    max_age = wapt.token_lifetime
                    token_content = token_gen.loads(token, max_age=max_age)
                    username = token_content.get('username', [])
                    usergroups = token_content.get('groups',[])
                    return (username,usergroups)
                except:
                    # password is not a token or token is invalid
                    pass

            # check if a valid Basic authorization header is decoded
            if not bearer == 'Basic':
                return (None,None)

            # decode Basic header
            (logon_name,password) = base64.b64decode(token.encode('ascii')+b'==').decode('utf8').split(':',1)

            # sanity check on logon name
            domain = ''
            if logon_name.count('\\') > 1 or logon_name.count('@') > 1 or (logon_name.count('\\') == 1 and logon_name.count('@') == 1):
                tasks_logger.warning("malformed logon credential : %s " % logon_name)
                return (None,None)

            tasks_logger.debug('service_auth_type : %s' % app.waptconfig.service_auth_type)

            # we need a scoped list of groups to check to avoid unecessary requests
            # groups are those defined in rules + waptselfservice + admin groups
            rules = None
            if wapt.is_enterprise():
                rules = common.self_service_rules(wapt)
                check_for_groups = list(rules.keys())
                check_for_groups.append('waptselfservice')
            else:
                check_for_groups=['waptselfservice']

            # auth through wapt server
            if app.waptconfig.service_auth_type == 'waptserver-ldap':
                if not wapt.is_enterprise():
                    raise Exception("waptagent-ldap for service auth type is an enterprise feature only")
                r =  wapt.self_service_auth(logon_name, password, wapt.host_uuid, check_for_groups)
                if r['result']['error']:
                    msg = 'Self service authentication failed for %s: %s' % (logon_name, r['result']['msg'])
                    raise EWaptAuthException(msg)
                if not r['result']['success']:
                    msg = 'WRONG_PASSWORD_USERNAME for %s' % logon_name
                    tasks_logger.warning(msg)
                    raise EWaptAuthException(msg)
                result = (logon_name , r['result']['groups'])
            # auth directly on a ldap server
            elif app.waptconfig.service_auth_type == 'waptagent-ldap':
                if not wapt.is_enterprise():
                    raise EWaptAuthException("waptagent-ldap for service auth type is an enterprise feature only")
                ad_result = auth_module_ad.check_credentials_ad(
                            conf = {'ldap_auth_server': app.waptconfig.ldap_auth_server,
                                    'ldap_auth_base_dn': app.waptconfig.ldap_auth_base_dn,
                                    "ad_domain_name" : app.waptconfig.ad_domain_name},
                            username = logon_name,
                            password = password,
                            list_group = check_for_groups)
                if ad_result['error']:
                    raise EWaptAuthException(ad_result['msg'])
                if not ad_result['success']:
                    raise EWaptAuthException('WRONG_PASSWORD_USERNAME for %s' % logon_name)
                result = (logon_name,ad_result['groups'])

            # auth with local system authentication (win32 or pam)
            elif (app.waptconfig.service_auth_type == 'system') or (password and app.waptconfig.service_auth_type in ('filetoken','nopassword') ) :  # basic auth fallback
                # extract domain from logon
                if '\\' in logon_name:
                    domain = logon_name.split('\\')[0]
                    username = logon_name.split('\\')[1]
                elif '@' in logon_name:
                    username = logon_name
                    domain = get_computer_domain()
                else:
                    username = logon_name

                if sys.platform == 'win32':
                    if domain == '' :
                        if in_case_insensitive_list(username, local_users()):
                            domain = '.'
                        else:
                            domain = get_computer_domain()

                    try:
                        huser = win32security.LogonUser(username, domain, password, win32security.LOGON32_LOGON_NETWORK_CLEARTEXT,getattr(win32security, app.waptconfig.logon32_provider))
                    except pywintypes.error as e:
                        if e.args[0] == 1326:
                            msg = 'WRONG_PASSWORD_USERNAME for %s\n%s' % (username,e.args[2])
                        else:
                            msg = e.args[2]
                        tasks_logger.warning(msg)
                        raise EWaptAuthException(msg)
                    usergroups = []
                    if in_case_insensitive_list(username, check_for_groups):
                        usergroups.append(username)
                    for group in check_for_groups:
                        if in_case_insensitive_list(group, usergroups):
                            continue
                        if common.check_is_member_of(huser, group):
                            usergroups.append(group)

                    # check admins
                    # check if user is domain admins or member of waptselfservice
                    if not 'waptselfservice' in usergroups:
                        try:
                            domain_admins_group = common.get_domain_admins_group_name()
                            if common.check_is_member_of(huser, domain_admins_group):
                                usergroups.append('waptselfservice')
                            else:
                                if not (app.waptconfig.waptservice_admin_filter):
                                    local_admin_group = common.get_local_admins_group_name()
                                    if common.check_is_member_of(huser, local_admin_group):
                                        usergroups.append('waptselfservice')
                        except Exception as e:
                            tasks_logger.warning('Fails to get groups of %s : %s' % (username,e,))

                    result = (username,usergroups)

                else:
                    usergroups = get_user_self_service_groups_unix(username, password, check_for_groups)
                    # add waptselfservice for admin
                    if not 'waptselfservice' in usergroups:
                        for group in ['root','sudo','wheel']:
                            if group in usergroups:
                                usergroups.append('waptselfservice')
                                break
                    result = (username,usergroups)
            else:
                return (None,None)

        tasks_logger.debug('check_auth_groups: %s' % (result,))
        return result
    except Exception as e:
        tasks_logger.warning('check_auth_groups: %s' % (e,))
        raise


def get_user_self_service_groups_unix(logon_name, password, check_for_groups = ['waptselfservice']):
    """Authenticate the user with pam, then returns all groups from check_groups to which
    the user belongs

    Returns:
        list of groups
    """
    p = pam.pam()
    success = p.authenticate(logon_name, password)
    if not success:
        raise EWaptAuthException('WRONG_PASSWORD_USERNAME')
    else:
        all_groups = []
        if logon_name in check_for_groups:
            all_groups.append(logon_name)
        user_groups = get_groups(logon_name)
        for group in check_for_groups:
            if group in user_groups:
                all_groups.append(group)
        return all_groups

@app.route('/lang/<language>')
def lang(language=None):
    session['lang'] = language
    return redirect('/')

@app.route('/ping')
@allow_local
def ping():
    if 'uuid' in request.args:
        w = scoped_wapt()
        data = dict(
            hostname=setuphelpers.get_hostname(),
            version=Version(__version__,3),
            version_full=WAPT_VERSION_FULL,
            uuid=w.host_uuid,
            waptserver=w.waptserver,
            tasks_count=app.task_manager.running_and_pending_count(),
        )
    else:
        data = dict(
            version=Version(__version__,3),
            version_full=WAPT_VERSION_FULL,
            tasks_count=app.task_manager.running_and_pending_count(),
        )
    return Response(jsondump(data), mimetype='application/json')

def write_token_file(username, data, secret,client_id=None):
    if not secret or len(secret)<64:
        raise Exception('Forbidden, no proper secret for token')
    if isinstance(data,str):
        data = data.encode('utf8')

    data = encrypt_token(data, secret)
    token_filepath = write_user_protected_file(generate_unique_string(), username, data)
    if not token_filepath:
        raise Exception('Forbidden, unable to find session or local profile for user %s' % username)
    return token_filepath

def encrypt_token(data: str,secret):
    data = waptlicences.encrypt_aes_pkcs7(data, secret)
    return data

def get_allowed_domain_usergroups(user_cn:str, check_for_groups:list) -> list:
    try:
        logger.debug('try to get domain groups for user %s' % user_cn)
        if sys.platform == 'win32':
            conn = pyldap.PyLdapClient()
        else:
            (hostname_from_keytab,domain_from_keytab) = get_domain_hostname_from_keytab()
            if domain_from_keytab:
                conn = pyldap.PyLdapClient(domain_name=domain_from_keytab)
            else:
                conn = pyldap.PyLdapClient()
        (ok, bound_domain_user) = conn.bind_sasl_kerberos()
        if ok:
            return conn.get_authorized_user_groups(user_cn, check_for_groups)
    except:
        return []

def local_login(wapt: Wapt, user: str, client_secret: str, token_gen=None):
    if not user or not client_secret or len(client_secret) < 64:
        time.sleep(3)
        return {}

    if not token_gen:
        token_gen = wapt.get_secured_token_generator(app.config['SECRET_KEY'])

    # check if user is a local admin or member of waptselfservice
    groups = []
    rules = common.self_service_rules(wapt)
    check_for_groups = list(rules.keys())

    if not 'waptselfservice' in check_for_groups:
        check_for_groups.append('waptselfservice')

    domain = None

    if '\\' in user:
        (domain,user) = user.split('\\',1)
    elif in_case_insensitive_list(user, local_users()):
        domain = '.'

    if domain != '.':
        domain_groups =  get_allowed_domain_usergroups(user, check_for_groups)
    else:
        domain_groups = []

    for group in check_for_groups:
        if user.lower()==group.lower():
            groups.append(group)
        elif domain_groups and in_case_insensitive_list(group, domain_groups):
            groups.append(group)
        elif user_is_member_of(user, group):
            groups.append(group)



    if not 'waptselfservice' in groups and (
            app.waptconfig.waptservice_allow_all_packages
            or
            ( sys.platform=='win32' and user.upper() == (os.environ.get('COMPUTERNAME')+'$').upper() ) # system account
            or
            ( sys.platform=='linux' and user == 'root' ) # root
            or
            ( not app.waptconfig.waptservice_admin_filter and (user_is_local_admin(user) or user_is_member_of(user, common.get_local_admins_group_name())))):
        groups.append('waptselfservice')

    auth_data = {'username': user, 'groups': groups}
    token = token_gen.dumps(auth_data)
    auth_data['token_filepath'] = write_token_file(user, token, secret=client_secret)
    # we only return the path to token in a location protected by target user system acls.
    return auth_data


@app.route('/login',methods=['GET','POST'])
@allow_local
def login():
    authenticated_username = None
    groups = []
    secret = None

    # we assume we need a valid Basic auth header here (token based auth is not valid)
    # this is like logout / login
    w = scoped_wapt()
    if request.authorization:
        try:
            auth_header = request.headers.get('Authorization') # either a Basic or a Bearer
            (authenticated_username,groups) = check_auth_groups(w,auth_header)
        except EWaptAuthException as e:
            if not app.waptconfig.service_auth_type in ('filetoken','nopassword'):
                time.sleep(3)
                return forbidden(msg=e)
        except Exception as e:
            if (sys.platform == 'win32') and isinstance(e,pywintypes.error):
                time.sleep(3)
                return forbidden(msg=e.strerror)
            time.sleep(3)
            return forbidden(msg='UNHANDLED_ERROR %s' % e.__class__)

    # generate a token for next requests
    token_gen = w.get_secured_token_generator(app.config['SECRET_KEY'])

    if request.method=='POST' and request.json.get('secret'):
        secret = request.json.get('secret')
    else:
        return forbidden()

    if not secret or len(secret) < 64:
        time.sleep(3)
        return forbidden(msg='invalid request')

    if not authenticated_username:
        # no user/password auth, but we can try to provide a token in a file
        if app.waptconfig.service_auth_type in ('filetoken','nopassword') and request.authorization:
            try:
                auth_data = local_login(w, request.authorization.username, secret, token_gen)
                auth_data['auth_type'] = app.waptconfig.service_auth_type
            except Exception as e:
                logger.warning('local_login failed for %s: %s' % (request.authorization.username, e))
                return forbidden(msg='WRONG_PASSWORD_USERNAME')
            if not auth_data:
                return forbidden(msg='WRONG_PASSWORD_USERNAME')
            logger.info('localtoken for user:%s, groups: %s generated.' % (auth_data.get('username'), auth_data.get('groups')))
            return Response(jsondump(auth_data), mimetype='application/json')
        else:
            time.sleep(3)
            return forbidden(msg='WRONG_PASSWORD_USERNAME')
    else:
        auth_data = {'username': authenticated_username, 'groups': groups}
        token = token_gen.dumps(auth_data)
        auth_data['token'] = base64.b64encode(encrypt_token(token,secret))
        auth_data['auth_type'] = app.waptconfig.service_auth_type
        logger.info('user %s authenticated. groups: %s' % (authenticated_username,groups))
        return Response(jsondump(auth_data), mimetype='application/json')

@app.route('/localtoken',methods=['POST'])
@allow_local
def localtoken():
    # create a token without groups for update/upgrade/events only
    w = scoped_wapt()
    if not request.authorization or not request.authorization.username:
        time.sleep(3)
        return forbidden()

    home_dir = user_home_directory(request.authorization.username)
    if not home_dir or not os.path.isdir(home_dir):
        time.sleep(3)
        return forbidden()

    secret = request.json.get('secret')
    if not secret or len(secret)<64:
        time.sleep(3)
        return forbidden()

    # generate a token for next requests
    token_gen = w.get_secured_token_generator(app.config['SECRET_KEY'])

    auth_data = {'username': request.authorization.username, 'auth_type':'profile'}
    token = token_gen.dumps(auth_data)
    token_filepath = write_token_file(request.authorization.username, token, secret)
    # we only return the path to token in a location protected by target user system acls.
    result = {'token_filepath': token_filepath}
    return Response(jsondump(result), mimetype='application/json')


##
def make_tasks_response(status='OK', tasks=[], msg=''):
    if not msg and not tasks:
        status='WARNING'
        msg=_('No task.')
    if not tasks is None and not isinstance(tasks,list):
        tasks=[tasks]
    return Response(jsondump(dict(tasks=tasks,status=status,msg=msg)), mimetype='application/json')

@app.route('/status')
@app.route('/status.json')
@allow_local_token
def status():
    rows = []
    with sqlite3.connect(app.waptconfig.dbpath) as con:
        try:
            con.row_factory = sqlite3.Row
            query = '''select s.package_uuid,s.package,s.version,s.install_date,
                                 s.install_status,s.install_output,r.icon_sha256sum,r.description,
                                 (select GROUP_CONCAT(p.version,"|") from wapt_package p where p.package=s.package) as repo_versions,
                                 explicit_by as install_par
                                 from wapt_localstatus s
                                 left join wapt_package r on r.package_uuid=s.package_uuid
                                 order by s.package'''
            cur = con.cursor()
            cur.execute(query)
            rows = []
            search = request.args.get('q', '')

            for row in cur.fetchall():
                pe = PackageEntry()
                rec_dict = dict((cur.description[idx][0], value) for idx, value in enumerate(row))
                for k in rec_dict:
                    setattr(pe, k, rec_dict[k])

                # hack to enable proper version comparison in templates
                pe.version = Version(pe.version)
                # calc most up to date repo version
                if pe.get('repo_versions', None) is not None:
                    pe.repo_version = max(Version(v) for v in pe.get('repo_versions', '').split('|'))
                else:
                    pe.repo_version = None

                if not search or pe.match_search(search):
                    rows.append(pe)

        except sqlite3.Error as e:
            logger.critical("*********** Error %s:" % e.args[0])

    return Response(jsondump(rows), mimetype='application/json')


def latest_only(packages):
    index = {}
    for p in sorted(packages, reverse=True):
        if not p.package in index:
            p.previous = []
            index[p.package] = p
        else:
            index[p.package].previous.append(p)

    return list(index.values())


@app.route('/keywords.json')
@app.route('/keywords')
@allow_local_token
def keywords():
    with sqlite3.connect(app.waptconfig.dbpath) as con:
        try:
            con.row_factory = sqlite3.Row
            cur = con.cursor()
            rows = []
            query = "select distinct trim(keywords) from wapt_package where keywords is not null and trim(keywords)<>''"
            rows = cur.execute(query)
            result = {}
            for k in rows:
                kws = k[0].lower().split(',')
                for kw in kws:
                    kwt = kw.strip().capitalize()
                    if not kwt in result:
                        result[kwt] = 1
                    else:
                        result[kwt] += 1
            return Response(jsondump(sorted(result.keys())), mimetype='application/json')
        except Exception as e:
            logger.critical('Error: %s' % e)
            return Response(jsondump([]), mimetype='application/json')


@app.route('/packages.json')
@app.route('/packages')
@app.route('/list')
@allow_local_token
def all_packages():
    w = scoped_wapt()
    all_sections = request.args.get('all_sections','0') == '1'
    limit = request.args.get('limit')
    if limit:
        limit=int(limit)

    # on enterprise, rules are needed to know if packages removal is allowed
    authorized = [p['package'] for p in usergroups_authorized_packages(w,session.usergroups, only_with_icons=False, all_sections=all_sections)]

    filtered = []
    if authorized:
        search = request.args.get('q', '').replace('\\', '')
        keywords = ensure_list(request.args.get('keywords', '').lower())

        all_packages = w.search(searchwords=search, exclude_host_repo=True, section_filter=None, newest_only=request.args.get('latest', '0') == '1', with_install_status=False, limit=limit)
        installed_status = {p['package']: p for p in w.waptdb.installed_status()}

        for package in all_packages:
            if package.package in installed_status:
                package.installed = installed_status[package.package]

            if len(keywords) > 0:
                match_kw = False
                package_keywords = ensure_list(package.keywords.lower())
                for kw in package_keywords:
                    if kw in keywords:
                        match_kw = True
                        break
                if not match_kw:
                    continue

            if package.package in authorized:
                filtered.append(package)

    for pe in filtered:
        # some basic search scoring
        score = 0
        if search in pe.package:
            score += 3
        if search in pe.description:
            score += 2
        pe.score = score

    filtered = sorted(filtered, key=lambda r: (r.score, r.signature_date, r.filename), reverse=True)
    return Response(jsondump(filtered), mimetype='application/json')

@app.route('/package_icon')
@allow_local_token
def package_icon():
    icon_sha256sum = sanitize_filename(request.args.get('icon_sha256sum'))
    icons_dir = os.path.join(app.waptconfig.wapt_base_dir,'cache','icons')
    if icon_sha256sum:
        icon_fn = '%s.png' % icon_sha256sum
        if os.path.isfile(os.path.join(icons_dir,icon_fn)):
            return send_from_directory(icons_dir,icon_fn)
        else:
            return send_from_directory(icons_dir,'unknown.png')
    else:
        return send_from_directory(icons_dir,'unknown.png')

def usergroups_authorized_packages(wapt:'Wapt', usergroups=[], only_with_icons=False, all_sections=False) -> list:
    """
    Returns:
        list[{'package','icon_sha256sum','repo'}]
    """
    rules = None
    result = []
    q = '''select distinct r.package, r.icon_sha256sum, r.repo from wapt_package r '''
    f = []
    if not all_sections:
        f.append('(not r.section in ("host","unit","profile","restricted","selfservice"))')
    if only_with_icons:
        f.append("(r.icon_sha256sum <> '')")
    if f:
        q = q + ' where ' + ' and '.join(f)
    all_packages = list(wapt.waptdb.query(q))

    if wapt.is_enterprise():
        rules = common.self_service_rules(wapt)

    for row in all_packages:
        if wapt.is_authorized_package_action('list', row['package'], usergroups, rules):
            result.append(row)

    return result

@app.route('/download_icons')
@app.route('/download_icons.json')
@allow_local_token
def download_icons():
    authorized_packages = usergroups_authorized_packages(scoped_wapt(), session.usergroups, only_with_icons=True)
    #return Response(jsondump(authorized_packages), mimetype='application/json')
    data = app.task_manager.add_task(WaptDownloadIcon(authorized_packages, created_by=session.username))
    if request.args.get('format', 'json') == 'json' or request.path.endswith('.json'):
        return Response(jsondump(data), mimetype='application/json')
    else:
        return Response('downloading %s icons' % len(authorized_packages), mimetype='text')


@app.route('/package_details')
@app.route('/package_details.json')
@allow_local_token
def package_details():
    # wapt=Wapt(config_filename=app.waptconfig.config_filename)
    package = request.args.get('package')
    authorized_packages = [ p['package'] for p in usergroups_authorized_packages(scoped_wapt(), session.usergroups, only_with_icons=False, all_sections=True) ]

    try:
        w = scoped_wapt()
        data = w.is_installed(package)
        if not data or not data['package'] in authorized_packages:
            data = [p for p in w.is_available(package) if p['package'] in authorized_packages]
            # keep only the newest...
            data = data and [data[-1].as_dict()]
        else:
            data = [data]
    except Exception as e:
        data = {'errors': [ensure_unicode(e)]}

    return Response(jsondump(dict(result=data, errors=[])), mimetype='application/json')

@app.route('/runstatus')
@allow_local_token
def get_runstatus():
    data = []
    with sqlite3.connect(app.waptconfig.dbpath) as con:
        con.row_factory = sqlite3.Row
        try:
            query = """select value,create_date from wapt_params where name='runstatus' limit 1"""
            cur = con.cursor()
            cur.execute(query)
            rows = cur.fetchall()
            data = [dict(ix) for ix in rows]
        except Exception as e:
            logger.critical("*********** error " + ensure_unicode(e))
    return Response(jsondump(data), mimetype='application/json')



def enqueue_task(task: WaptTask) -> dict:
    task.created_by = session.username
    force = request.args.get('force')
    if force is not None:
        task.force = int(force) != 0
    priority = request.args.get('priority', None)
    if priority is not None:
        task.priority = int(priority)
    notify_user = request.args.get('notify_user')
    if notify_user is not None:
        task.notify_user = int(notify_user) != 0
    else:
        task.notify_user = waptconfig.notify_user
    notify_server_on_finish = request.args.get('notify_server')
    if notify_server_on_finish is not None:
        task.notify_server_on_finish = int(notify_server_on_finish) != 0
    start_not_before = request.args.get('start_not_before', None)
    if start_not_before is not None:
        task.start_not_before = start_not_before
    start_not_after = request.args.get('start_not_after', None)
    if start_not_after is not None:
        task.start_not_after = start_not_after
    always_replace = int(request.args.get('always_replace', '0')) != 0
    always_append = int(request.args.get('always_append', '0')) != 0
    return app.task_manager.add_task(task, always_replace=always_replace, always_append=always_append).as_dict()


@app.route('/checkupgrades')
@app.route('/checkupgrades.json')
@allow_local_token
def get_checkupgrades():
    with sqlite3.connect(app.waptconfig.dbpath) as con:
        con.row_factory = sqlite3.Row
        data = ""
        try:
            query = """select * from wapt_params where name="last_update_status" limit 1"""
            cur = con.cursor()
            cur.execute(query)
            row = cur.fetchone()
            if row:
                data = json.loads(row['value'])
                # update runing_tasks.
                if app.task_manager:
                    with app.task_manager.status_lock:
                        if app.task_manager.running_task:
                            data['running_tasks'] = [app.task_manager.running_task.as_dict()]
                        else:
                            data['running_tasks'] = []
                        data['pending_tasks'] = [task.as_dict() for task in sorted(app.task_manager.tasks_queue.queue)]

                # if enterprise, add waptwua status
                query = """select * from wapt_params where name="waptwua.status" limit 1"""
                cur = con.cursor()
                cur.execute(query)
                row = cur.fetchone()
                if row:
                    data['wua_status'] = row['value']
                    # check count of updates
                    try:
                        query = """select * from wapt_params where name="waptwua.updates_localstatus" limit 1"""
                        cur = con.cursor()
                        cur.execute(query)
                        row = cur.fetchone()
                        if row:
                            wua_localstatus = json.loads(row['value'])
                            wua_pending_count = len([u['update_id'] for u in wua_localstatus if u['status'] == 'PENDING'])
                            data['wua_pending_count'] = wua_pending_count
                    except Exception as e:
                        logger.critical('Unable to read waptwua updates_localstatus from DB: %s' % e)
            else:
                data = {}

        except Exception as e:
            logger.critical("*********** error %s" % (ensure_unicode(e)))
    return Response(jsondump(data), mimetype='application/json')

@app.route('/waptservicerestart')
@app.route('/waptservicerestart.json')
@allow_local_token
def waptservicerestart():
    """Restart local waptservice"""
    data = enqueue_task(WaptServiceRestart())
    return make_tasks_response(tasks=data,msg=_('Restart of waptservice enqueued'),status='OK')

@app.route('/reload_config')
@app.route('/reload_config.json')
@allow_local_token
def reload_config():
    """trigger reload of wapt-get.ini file for the service"""
    data = enqueue_task(WaptNetworkReconfig())
    return make_tasks_response(tasks=data, msg=_('Reload of waptservice configuration enqueued'), status='OK')

@app.route('/upgrade')
@app.route('/upgrade.json')
@allow_local_token
def upgrade():
    update_packages = int(request.args.get('update', '1')) != 0

    only_priorities = None
    if 'only_priorities' in request.args:
        only_priorities = ensure_list(request.args.get('only_priorities', None), allow_none=True)
    only_if_not_process_running = int(request.args.get('only_if_not_process_running', '0')) != 0

    all_tasks = []
    msg = []

    if update_packages:
        all_tasks.append(enqueue_task(WaptUpdate()))
        msg.append('Update')

    task = WaptUpgrade(only_priorities=only_priorities, only_if_not_process_running=only_if_not_process_running)
    all_tasks.append(enqueue_task(task))
    msg.append('Upgrade')
    all_tasks.append(enqueue_task(WaptCleanup()))
    if scoped_wapt().is_enterprise() and setuphelpers.get_loggedinusers():
        all_tasks.append(enqueue_task(WaptRunSessionSetup()))
        msg.append('Session-setup')

    return make_tasks_response(tasks=all_tasks, msg=', '.join(msg) +' ' +_('tasks enqueued'),status='OK')

@app.route('/waptwua_scan')
@allow_local_token
def waptwua_scan():
    all_tasks = []
    if sys.platform == 'win32' and scoped_wapt().is_enterprise():
        all_tasks.append(enqueue_task(WaptWUAScanTask()))
    return make_tasks_response(tasks=all_tasks,msg=_('Scan of required Windows system updates enqueued'),status='OK')

@app.route('/waptwua_download')
@allow_local_token
def waptwua_download():
    all_tasks = []
    if sys.platform == 'win32' and scoped_wapt().is_enterprise() and scoped_wapt().waptwua_enabled:
        all_tasks.append(enqueue_task(WaptWUADowloadTask()))
    return make_tasks_response(tasks=all_tasks,msg=_('Download of pending Windows system updates enqueued'),status='OK')

@app.route('/waptwua_install')
@allow_local_token
def waptwua_install():
    all_tasks = []
    if sys.platform == 'win32' and scoped_wapt().is_enterprise() and scoped_wapt().waptwua_enabled:
      all_tasks.append(enqueue_task(WaptWUAInstallTask()))
    return make_tasks_response(tasks=all_tasks,msg=_('Install of pending Windows Updates enqueued'),status='OK')

@app.route('/download_upgrade')
@app.route('/download_upgrade.json')
@allow_local_token
def download_upgrade():
    update_packages = int(request.args.get('update', '0')) != 0
    data = []
    if update_packages:
        data.append(enqueue_task(WaptUpdate()))
    data.append(enqueue_task(WaptDownloadUpgrade()))
    return make_tasks_response(tasks=data, msg=_('Download of pending package upgrades enqueued'),status='OK')

@app.route('/update')
@app.route('/update.json')
@allow_local_token
def update():
    data = []
    data.append(enqueue_task(WaptUpdate()))
    return make_tasks_response(tasks=data, msg=_('Update of packages index enqueued'),status='OK')

@app.route('/audit')
@app.route('/audit.json')
@allow_local_token
def audit():
    w = scoped_wapt()
    if not w.is_enterprise():
        return
    tasks = []
    force = int(request.args.get('force', '0')) != 0

    now = datetime2isodate(datetime.datetime.utcnow())

    packagenames = ensure_list(request.args.get('package', None), allow_none=True)
    if packagenames is None:
        packagenames = []
        for package_status in w.installed():
            if force or not package_status.next_audit_on or (now >= package_status.next_audit_on):
                packagenames.append(package_status.package)

    authorized_packages = []
    if packagenames:
        # on enterprise, rules are needed to know if packages removal is allowed
        rules = None
        if w.is_enterprise():
            rules = common.self_service_rules(w)

        for apackage in packagenames:
            if w.is_authorized_package_action('audit', apackage, session.usergroups, rules):
                authorized_packages.append(apackage)

        if authorized_packages:
            task = WaptAuditPackage(authorized_packages)
            tasks.append(enqueue_task(task))

            return make_tasks_response(status='OK', tasks=tasks, msg=_('Audit task queued for %s package(s)') % len(authorized_packages))

    return make_tasks_response(status='WARNING', tasks=tasks, msg=_('No package matching the authorization rules'))

@app.route('/update_status')
@app.route('/update_status.json')
@allow_local_token
def update_status():
    data = enqueue_task(WaptUpdateServerStatus())
    return make_tasks_response(status='OK', tasks=data, msg=_("Update of the Agent's status to server enqueued"))

@app.route('/longtask')
@app.route('/longtask.json')
@allow_local_token
def longtask():
    data = enqueue_task(
        WaptLongTask(
            duration=int(request.args.get('duration', '60')),
            raise_error=int(request.args.get('raise_error', 0))
            )
        )
    return make_tasks_response(status='OK', tasks=data, msg=_("Long task (for test) enqueued"))

@app.route('/cleanup')
@app.route('/cleanup.json')
@app.route('/clean')
@allow_local_token
def cleanup():
    data = enqueue_task(WaptCleanup())
    return make_tasks_response(status='OK', tasks=data, msg=_("Cleanup of unuseful cached packages enqueued"))

@app.route('/register')
@app.route('/register.json')
@allow_local_token
def register():
    data = enqueue_task(WaptRegisterComputer())
    return make_tasks_response(status='OK', tasks=data, msg=_("Registration of agent to the server enqueued"))

@app.route('/inventory')
@app.route('/inventory.json')
@allow_local_token
def inventory():
    data = scoped_wapt().inventory()
    return Response(jsondump(data), mimetype='application/json')

@app.route('/install', methods=['GET'])
@app.route('/install.json', methods=['GET'])
@allow_local_token
def install():
    package_requests = request.args.get('package')
    if not isinstance(package_requests, list):
        package_requests = [package_requests]

    only_priorities = None
    if 'only_priorities' in request.args:
        only_priorities = ensure_list(request.args.get('only_priorities', None), allow_none=True)
    only_if_not_process_running = int(request.args.get('only_if_not_process_running', '0')) != 0

    w = scoped_wapt()
    # on enterprise, rules are needed to know if packages removal is allowed
    rules = None
    if w.is_enterprise():
        rules = common.self_service_rules(w)

    authorized_packages = []
    for apackage in package_requests:
        if w.is_authorized_package_action('install', apackage, session.usergroups, rules):
            authorized_packages.append(apackage)
        else:
            return authenticate()

    logger.info("user %s authenticated" % session.username)
    data = []
    for apackage in authorized_packages:
        data.append(enqueue_task(WaptPackageInstall(
                    apackage,
                    only_priorities=only_priorities,
                    only_if_not_process_running=only_if_not_process_running
                    )))

    if authorized_packages and w.is_enterprise():
        data.append(enqueue_task(WaptAuditPackage(packagenames=authorized_packages, priority=100)))
        if setuphelpers.get_loggedinusers():
            data.append(enqueue_task(WaptRunSessionSetup()))

    if authorized_packages:
        return make_tasks_response(status='OK', tasks=data, msg=_("Install of %s enqueued") % ','.join(authorized_packages))
    else:
        return make_tasks_response(status='WARNING', tasks=data, msg=_("No package install task enqueued"))

@app.route('/download', methods=['GET'])
@app.route('/download.json', methods=['GET'])
@allow_local_token
def download():
    package_requests = request.args.get('package')
    if not isinstance(package_requests, list):
        package_requests = [package_requests]

    w = scoped_wapt()
    # on enterprise, rules are needed to know if packages removal is allowed
    rules = None
    if w.is_enterprise():
        rules = common.self_service_rules(w)

    authorized_packages = []
    for apackage in package_requests:
        if w.is_authorized_package_action('install', apackage, session.usergroups, rules):
            authorized_packages.append(apackage)
        else:
            return authenticate()

    logger.info("user %s authenticated" % session.username)
    data = []
    if authorized_packages:
        for apackage in authorized_packages:
            data.append(enqueue_task(WaptDownloadPackage(authorized_packages)))
        return make_tasks_response(status='OK', tasks=data, msg=_("Download of %s enqueued") % ','.join(authorized_packages))
    else:
        return make_tasks_response(status='WARNING', tasks=data, msg=_("No package download task enqueued"))


@app.route('/remove', methods=['GET'])
@app.route('/remove.json', methods=['GET'])
@app.route('/remove.html', methods=['GET'])
@allow_local_token
def remove():
    logger.info("user %s authenticated" % session.username)

    package_requests = request.args.get('package')
    if not isinstance(package_requests, list):
        package_requests = [package_requests]

    only_priorities = None
    if 'only_priorities' in request.args:
        only_priorities = ensure_list(request.args.get('only_priorities', None), allow_none=True)
    only_if_not_process_running = int(request.args.get('only_if_not_process_running', '0')) != 0

    data = []
    w = scoped_wapt()

    authorized_packages = []
    for apackage in package_requests:
        installed = w.is_installed(apackage)
        if not installed:
            continue

        if only_priorities is not None and not installed.get('priority') in only_priorities:
            continue

        if w.remove_is_allowed(apackage, session.usergroups):
            authorized_packages.append(apackage)
            data.append(enqueue_task(
                WaptPackageRemove(apackage, only_if_not_process_running=only_if_not_process_running, only_priorities=only_priorities)))

    if data:
        return make_tasks_response(status='OK', tasks=data, msg=_("Removal of %s enqueued") % ','.join(authorized_packages))
    else:
        return make_tasks_response(status='WARNING', tasks=data, msg=_("No package removal task enqueued"))


@app.route('/forget', methods=['GET'])
@app.route('/forget.json', methods=['GET'])
@allow_local_token
def forget():
    packages = request.args.get('package')
    if not isinstance(packages, list):
        packages = [packages]
    logger.info("Forget package(s) %s" % packages)

    data = []
    w = scoped_wapt()

    authorized_packages = []
    for apackage in packages:
        installed = w.is_installed(apackage)
        if not installed:
            continue
        if w.remove_is_allowed(apackage, session.usergroups):
            authorized_packages.append(apackage)
            data.append(enqueue_task(
                WaptPackageForget(apackage)))

    if data:
        return make_tasks_response(status='OK', tasks=data, msg=_("Forget of %s enqueued") % ','.join(authorized_packages))
    else:
        return make_tasks_response(status='WARNING', tasks=data, msg=_("No package forget task enqueued"))

@app.route('/is_enterprise', methods=['GET'])
@allow_local
def is_enterprise():
    result = scoped_wapt().is_enterprise()
    return Response(jsondump(result), mimetype='application/json')


@app.route('/tasks')
@app.route('/tasks.json')
@allow_local_token
def tasks():
    last_read_event_id = int(request.args.get('last_read_event_id') or '-1')
    timeout = int(request.args.get('timeout') or '-1')

    data = None
    start_time = time.time()
    status = 200

    while True:
        # wait for events manager initialisation
        if app.task_manager.events:
            actual_last_event_id = app.task_manager.events.last_event_id()
            if actual_last_event_id is not None and actual_last_event_id <= last_read_event_id:
                if (time.time() - start_time) * 1000 > timeout:
                    status = 204
                    break
            elif actual_last_event_id is None or actual_last_event_id > last_read_event_id:
                status = 200
                data = app.task_manager.tasks_status()
                break
        if (time.time() - start_time) * 1000 > timeout:
            break

        # avoid eating cpu
        time.sleep(0.1)

    return Response(jsondump(data), mimetype='application/json', status=status)

@app.route('/tasks_status')
@app.route('/tasks_status.json')
@allow_local_token
def tasks_status():
    last_read_event_id = int(request.args.get('last_read_event_id') or '-1')
    timeout = int(request.args.get('timeout') or '-1')
    max_events_count = int(request.args.get('max_events_count') or '0')

    result = {}
    start_time = time.time()
    data = None
    status = 200

    while True:
        if app.task_manager.events:
            actual_last_event_id = app.task_manager.events.last_event_id()
            result['last_event_id'] = actual_last_event_id
            if actual_last_event_id is not None and actual_last_event_id <= last_read_event_id:
                if (time.time() - start_time) * 1000 > timeout:
                    status = 204
                    break
            elif actual_last_event_id is None or actual_last_event_id > last_read_event_id:
                data = app.task_manager.tasks_status()
                status = 200
                break

        if (time.time() - start_time) * 1000 > timeout:
            break

        # avoid eating cpu
        time.sleep(0.2)

    if data:
        tasks = []
        tasks.extend(data['pending'])
        if data['running']:
            tasks.append(data['running'])
        tasks.extend(data['done'])
        tasks.extend(data['errors'])
        tasks.extend(data['cancelled'])
        result['tasks'] = tasks

    if max_events_count:
        result['events'] = app.task_manager.events.get_missed(last_read=last_read_event_id, max_count=max_events_count, owner=session.username)

    return Response(jsondump(result), mimetype='application/json', status=status)

@app.route('/task')
@app.route('/task.json')
@allow_local_token
def task():
    id = int(request.args.get('id') or '-1')
    task = {}
    if id >= 0:
        tasks = app.task_manager.tasks_status()
        all_tasks = tasks['done']+tasks['pending']+tasks['errors']
        if tasks['running']:
            all_tasks.append(tasks['running'])
        all_tasks = [task for task in all_tasks if task and task['id'] == id]
        if all_tasks:
            task = all_tasks[0]

    return Response(jsondump(task), mimetype='application/json')

@app.route('/cancel_all_tasks')
@app.route('/cancel_all_tasks.json')
@allow_local_token
def cancel_all_tasks():
    data = app.task_manager.cancel_all_tasks()
    return Response(jsondump(data), mimetype='application/json')

@app.route('/cancel_running_task')
@app.route('/cancel_running_task.json')
@allow_local_token
def cancel_running_task():
    data = app.task_manager._wapt.task_is_cancelled.set()
    if request.args.get('format', 'json') == 'json' or request.path.endswith('.json'):
        return Response(jsondump(data), mimetype='application/json')
    else:
        return render_wapt_template('default.html', data=data)


@app.route('/cancel_task')
@app.route('/cancel_task.json')
@allow_local_token
def cancel_task():
    id = int(request.args.get('id') or '-1')
    data = None
    if id >= 0:
        data = app.task_manager.cancel_task(id)
    return Response(jsondump(data), mimetype='application/json')

@app.route('/events')
@app.route('/events.json')
@allow_local_token
def events():
    """Get the last waptservice events.
    Blocking call for timeout seconds.

    Args:
        last_read_event_id (int): id of last read event.
        timeout (float): time to wait until new events come in
    """
    last_read_event_id = int(request.args.get('last_read_event_id', session.get('last_read_event_id', '0')))
    timeout = int(request.args.get('timeout', '10000'))
    max_count = int(request.args.get('max_events_count', '0')) or None
    if app.task_manager.events:
        data = app.task_manager.events.get_missed(last_read=last_read_event_id, max_count=max_count, owner=session.username)
        if not data and timeout > 0.0:
            start_time = time.time()
            while not data and (time.time() - start_time) * 1000 <= timeout:
                time.sleep(1.0)
                data = app.task_manager.events.get_missed(last_read=last_read_event_id, max_count=max_count, owner=session.username)
            if data:
                session['last_read_event_id'] = data[-1].id
    else:
        data = None
    return Response(jsondump(data), mimetype='application/json')


@app.route('/polldebug.json',methods=['POST'])
@allow_local
def polldebug():
    return Response(jsondump(request.json), mimetype='application/json')

class WaptTaskManager(threading.Thread):
    def __init__(self, config_filename='c:/wapt/wapt-get.ini', wapt_base_dir=None):
        threading.Thread.__init__(self)
        self.name = 'WaptTaskManager'
        self.status_lock = threading.RLock()
        self._wapt = None
        self.tasks = []

        self.tasks_queue = queue.PriorityQueue()
        self.tasks_counter = 0

        self.tasks_done = []
        self.tasks_error = []
        self.tasks_cancelled = []
        self.events = None

        self.running_task = None
        self.config_filename = config_filename
        self.wapt_base_dir = wapt_base_dir

        self.last_update_server_date = None

        self.last_upgrade = None
        self.last_update = None
        self.last_audit = None
        self.last_sync = None
        self.last_forced_installs = None

        self.logger = logging.getLogger('wapttasks')


    @property
    def wapt(self) -> Wapt:
        if not self._wapt:
            self._wapt = Wapt(config_filename=self.config_filename, merge_config_packages=True, wapt_base_dir = self.wapt_base_dir)
            self._wapt.threadid = threading.current_thread().ident
        if self._wapt.threadid != threading.current_thread().ident:
            raise Exception('Access to task_manager Wapt instance from a foreign Thread')
        return self._wapt

    def setup_event_queue(self) -> WaptEvents:
        self.events = WaptEvents()
        return self.events

    def update_runstatus(self, status):
        # update database with new runstatus
        if status != self.wapt.runstatus:
            self.wapt.runstatus = status
            if self.events:
                # dispatch event to listening parties
                self.events.post_event("STATUS", self.wapt.get_last_update_status(), owner='SYSTEM')

    def update_server_status(self):
        try:
            if not self.wapt.waptserver:  # starter edition
                return
            result = self.wapt.update_server_status()
            if result and result.get('success') and result['result']['uuid']:
                self.last_update_server_date = datetime.datetime.utcnow()
            elif result and not result.get('success'):
                self.logger.critical('Unable to update server status: %s' % result['msg'])
            else:
                self.logger.info('Unable to update server status: No answer')
        except Exception as e:
            self.logger.info('Unable to update server status: %s' % repr(e))

    def broadcast_tasks_status(self, event_type, task):
        """event_type : TASK_ADD TASK_START TASK_STATUS TASK_FINISH TASK_CANCEL TASK_ERROR
        """
        if self.events and task:
            self.events.post_event(event_type, task.as_dict(), owner=task.created_by)

    def add_task(self, task, always_replace=False, always_append=False) -> WaptTask:
        """Adds a new WaptTask for processing"""
        with self.status_lock:
            if always_replace:
                same = None
                to_cancel = []
                for pending in self.tasks_queue.queue:
                    if pending.same_action(task):
                        to_cancel.append(task.id)

                if to_cancel:
                    self.cancel_tasks(to_cancel)

            else:
                same = [pending for pending in self.tasks_queue.queue if pending.same_action(task)]
                if self.running_task and self.running_task.same_action(task):
                    same.append(self.running_task)

            # keep track of last update/upgrade add date to avoid relaunching
            if isinstance(task, WaptUpdate):
                self.last_update = datetime.datetime.utcnow()
            if isinstance(task, WaptUpgrade):
                self.last_upgrade = datetime.datetime.utcnow()
            if isinstance(task, WaptSyncRepo):
                self.last_sync = datetime.datetime.utcnow()

            # not already in pending  actions...
            if not same or always_append or always_replace:
                self.logger.info('Add task %s to queue' % (task,))

                task.task_manager = self

                self.tasks_counter += 1
                task.id = self.tasks_counter
                # default order is task id
                task.order = self.tasks_counter
                self.tasks_queue.put(task)
                self.tasks.append(task)
                self.broadcast_tasks_status('TASK_ADD', task)
                return task
            else:
                self.logger.debug('Not adding task %s. Already a same action in queue' % (task,))
                return same[0]

    def check_configuration(self) -> bool:
        """Check wapt configuration, reload ini file if changed"""
        try:
            self.logger.debug("Checking if config file has changed")
            if self.wapt.reload_config_if_updated():
                self.logger.info("Core Wapt config file has changed, reloaded. Reloading waptservice config too")
                waptconfig.load()
                return True
            return False

        except Exception as e:
            self.logger.critical('Error reloading configuration: %s' % ensure_unicode(e))
            return False

    def run_scheduled_audits(self):
        """Add packages audit tasks to the queue"""
        now = datetime2isodate(datetime.datetime.utcnow())

        self.last_audit = datetime.datetime.utcnow()

        packages = [ pe['package'] for pe in self.wapt.waptdb.query("""select
                package,next_audit_on
            from wapt_localstatus
            where install_status = 'OK' and
                (next_audit_on='' or next_audit_on is null or next_audit_on <= '%s') and
                (next_audit_on > last_audit_on or last_audit_on is null or last_audit_on='')
            order by next_audit_on
            """ % now)]

        if packages:
            task = WaptAuditPackage(packages, created_by='SCHEDULER')
            task.notify_server_on_finish=True
            self.add_task(task)

    @property
    def last_waptwua_download(self) -> datetime.datetime:
        return self.wapt.read_param('last_waptwua_download', ptype='datetime')

    @last_waptwua_download.setter
    def last_waptwua_download(self, value: datetime.datetime):
        if value is None:
            self.wapt.delete_param('last_waptwua_download')
        else:
            self.wapt.write_param('last_waptwua_download', value)

    @property
    def last_waptwua_install(self) -> datetime.datetime:
        return self.wapt.read_param('last_waptwua_install', ptype='datetime')

    @last_waptwua_install.setter
    def last_waptwua_install(self, value: datetime.datetime):
        if value is None:
            self.wapt.delete_param('last_waptwua_install')
        else:
            self.wapt.write_param('last_waptwua_install', value)

    def run_immediate_installs(self,force=False,only_priorities=None, only_if_not_process_running=False):
        self.logger.debug('Check if package should be forcibly installed')
        last_update_status = self.wapt.get_last_update_status()
        immediate_install = last_update_status.get('pending',{}).get('immediate_installs',[])
        if immediate_install:
            for package in immediate_install:
                self.add_task(WaptPackageInstall(package, force=force, created_by='SCHEDULER',
                    only_priorities=only_priorities, only_if_not_process_running=only_if_not_process_running, notify_server_on_finish=True))


    def _can_run_auto_upgrade(self):
        if setuphelpers.running_on_ac():
            return True

        if hasattr(psutil,"sensors_battery"):
            try:
                battery = psutil.sensors_battery()
                if hasattr(battery,"power_plugged"):
                    if not battery.power_plugged :
                        if battery.percent < waptconfig.minimum_battery_percent:
                            logger.info(_('auto upgrade: Battery level too low %s, minimum battery %s' % (battery.percent,waptconfig.minimum_battery_percent)))
                            return False
            except:
                return True

    def check_scheduled_tasks(self):
        """Add update/upgrade tasks if elapsed time since last update/upgrade is over"""
        self.logger.debug('Check scheduled tasks')

        if datetime.datetime.utcnow() - self.start_time >= datetime.timedelta(days=1):
            self.start_time = datetime.datetime.utcnow()
            self.add_task(WaptServiceRestart(created_by='DAILY RESTART'))

        if waptconfig.waptupdate_task_period is not None:
            if self.last_update is None or \
                    (datetime.datetime.utcnow() - self.last_update) > get_time_delta(waptconfig.waptupdate_task_period, 'm') or \
                    (setuphelpers.datetime2isodate() > self.wapt.read_param('next_update_on', '9999-12-31')):
                try:
                    self.wapt.update()
                    if waptconfig.download_after_update_with_waptupdate_task_period:
                        reqs = self.wapt.check_downloads()
                        for req in reqs:
                            self.add_task(WaptDownloadPackage(req.asrequirement(), notify_user=True, created_by='SCHEDULER'))
                    self.add_task(WaptUpdate(notify_user=False, notify_server_on_finish=True, created_by='SCHEDULER'))
                except Exception as e:
                    self.logger.critical('Error for update in check_scheduled_tasks: %s' % e)

        if waptconfig.waptupgrade_task_period is not None and self._can_run_auto_upgrade():
            if self.last_upgrade is None or (datetime.datetime.utcnow() - self.last_upgrade) > get_time_delta(waptconfig.waptupgrade_task_period, 'm'):
                try:
                    self.add_task(WaptUpgrade(notify_user=False, created_by='SCHEDULER', only_if_not_process_running=True, notify_server_on_finish=True))
                except Exception as e:
                    self.logger.warning('Error for upgrade in check_scheduled_tasks: %s' % e)
                self.add_task(WaptCleanup(notify_user=False, created_by='SCHEDULER'))
                if self.wapt.is_enterprise() and setuphelpers.get_loggedinusers():
                    self.add_task(WaptRunSessionSetup())

        # we need next_audit before forced install to be sure trigger pending forced installs and not run outdated audit scripts
        next_audit = self.wapt.get_next_audit_datetime()
        if not next_audit:
            if self.last_audit is None:
                next_audit = datetime.datetime.utcnow()
            else:
                next_audit = self.last_audit + get_time_delta(self.wapt.waptaudit_task_period)

        if (self.last_forced_installs is None or
            (datetime.datetime.utcnow() - self.last_forced_installs) > get_time_delta(waptconfig.forced_installs_task_period, 'm') or
            (next_audit and datetime.datetime.utcnow() >= next_audit)
            ):
            self.last_forced_installs = datetime.datetime.utcnow()
            self.run_immediate_installs()

        if next_audit and datetime.datetime.utcnow() >= next_audit:
            try:
                self.run_scheduled_audits()
            except Exception as e:
                self.logger.critical('Error checking audit: %s' % e)

        if waptconfig.enable_remote_repo and self.wapt.is_enterprise() and not(waptconfig.sync_only_forced):
            if (self.last_sync is None or (datetime.datetime.utcnow() - self.last_sync > get_time_delta(waptconfig.local_repo_sync_task_period, 'm'))) and ((waptconfig.local_repo_time_for_sync_start is None) or is_between_two_times(waptconfig.local_repo_time_for_sync_start, waptconfig.local_repo_time_for_sync_end)):
                try:
                    self.add_task(WaptSyncRepo(notify_user=False, created_by='SCHEDULER', socketio_client=sio.socketio_client))
                except Exception as e:
                    self.logger.critical('Error syncing local repo with server repo : %s' % e)

        if sys.platform == 'win32' and self.wapt.is_enterprise() and self.wapt.waptwua_enabled:
            uptime_minutes = win32api.GetTickCount() / 60 / 1000
            if waptconfig.waptwua_install_scheduling:
                # delayed 10 minutes after boot
                if uptime_minutes > get_time_delta(waptconfig.waptwua_postboot_delay).total_seconds()/60:
                    if self.last_waptwua_install is None or (datetime.datetime.utcnow() - self.last_waptwua_install > get_time_delta(waptconfig.waptwua_install_scheduling, 'd')):
                        #if self.wapt.read_param('waptwua.status', '') == 'PENDING_UPDATES':
                        self.add_task(WaptWUAInstallTask(notify_user=False, notify_server_on_finish=True, created_by='SCHEDULER'))
                        self.last_waptwua_install = datetime.datetime.utcnow()
                        self.last_waptwua_download = datetime.datetime.utcnow()

            if waptconfig.waptwua_download_scheduling:
                # .layed 10 minutes after boot
                if uptime_minutes > get_time_delta(waptconfig.waptwua_postboot_delay).total_seconds()/60:
                    if self.last_waptwua_download is None or (datetime.datetime.utcnow() - self.last_waptwua_download > get_time_delta(waptconfig.waptwua_download_scheduling, 'd')):
                        self.add_task(WaptWUADowloadTask(notify_user=False, notify_server_on_finish=True, created_by='SCHEDULER'))
                        self.last_waptwua_download = datetime.datetime.utcnow()

    def run(self):
        """Queue management, event processing"""
        if sys.platform == 'win32':
            try:
                pythoncom.CoInitializeEx(pythoncom.COINIT_MULTITHREADED)
            except pythoncom.com_error:
                # already initialized.
                pass

        self.start_time = datetime.datetime.utcnow()
        self.setup_event_queue()

        self.logger.info('Wapt tasks management initialized with {} configuration'.format(self.config_filename))
        self.logger.info('Core Configuration: %s' % self.wapt.as_dict())

        if waptconfig.reconfig_on_network_change:
            self.start_network_monitoring()
            self.start_ipaddr_monitoring()

        self.logger.info("Wapt tasks queue started")
        gc.set_debug(gc.DEBUG_UNCOLLECTABLE)

        while True:
            try:
                need_sleep = False
                # check wapt configuration, reload ini file if changed
                # reload wapt config
                self.check_configuration()
                # force update if host capabilities have changed
                new_capa = self.wapt.host_capabilities_fingerprint()
                old_capa = self.wapt.read_param('host_capabilities_fingerprint')
                last_update_config_fingerprint = self.wapt.read_param('last_update_config_fingerprint')
                if old_capa != new_capa or last_update_config_fingerprint != self.wapt.merged_config_hash:
                    if old_capa != new_capa:
                        self.logger.debug('Host capabilities have changed since last update %s -> %s, forcing update' % (old_capa,new_capa))
                    if last_update_config_fingerprint != self.wapt.merged_config_hash:
                        self.logger.debug('Service config have changed %s -> %s, forcing update' % (last_update_config_fingerprint,self.wapt.merged_config_hash))
                    # force update
                    self.last_update = None

                # check tasks queue
                next_task = self.tasks_queue.get(timeout=waptconfig.waptservice_poll_timeout)
                try:
                    nowutc = datetime2isodate()
                    if (next_task.start_not_before and nowutc < next_task.start_not_before):
                        # requeue...
                        self.logger.debug('Tasks "%s" is not yet ready to start. not before %s' % (next_task,next_task.start_not_before))
                        need_sleep = self.tasks_queue.empty() or not [t for t in self.tasks_queue.queue if not t.start_not_before or nowutc >= next_task.start_not_before]
                        if need_sleep:
                            self.logger.debug('only non ready tasks in queue; need sleep')
                        self.tasks_queue.put(next_task)

                        continue

                    if (next_task.start_not_after and nowutc < next_task.start_not_after):
                        self.logger.info('Tasks "%s" is too old. Canceled. not after %s' % (next_task,next_task.start_not_after))
                        self.tasks_cancelled.append(next_task)
                        continue

                    self.running_task = next_task

                    self.logger.info('Running task "%s" created by "%s"' % (self.running_task,self.running_task.created_by))
                    # don't send update_run status for updatestatus itself...
                    self.broadcast_tasks_status('TASK_START', self.running_task)
                    if self.running_task.notify_server_on_start:
                        self.update_runstatus(_('Running "{description}"').format(description=self.running_task))
                        self.update_server_status()
                    try:
                        def update_running_status(append_line=None, set_status=None):
                            if append_line:
                                self.wapt.runstatus = append_line

                                if self.running_task:
                                    self.running_task.logs.append(append_line)

                                if self.events:
                                    self.events.post_event('PRINT', ensure_unicode(append_line), owner='SYSTEM')

                                if self.events and self.running_task:
                                    self.broadcast_tasks_status('TASK_STATUS', self.running_task)

                        with LogOutput(console=sys.stderr, update_status_hook=update_running_status):
                            try:
                                self.running_task.run()
                            except Exception as e:
                                print(e)
                                raise


                        if self.running_task:
                            running = self.running_task
                            self.running_task = None
                            self.tasks_done.append(running)

                            self.broadcast_tasks_status('TASK_FINISH', running)
                            if running.notify_server_on_finish:
                                self.update_runstatus(_('Done: {description}\n{summary}').format(description=running, summary=running.summary))
                                self.update_server_status()

                    except common.EWaptCancelled as e:
                        self.logger.info('Task cancelled %s' % (self.running_task,))
                        if self.running_task:
                            self.running_task.logs.append("{}".format(ensure_unicode(e)))
                            self.running_task.summary = _("Canceled")
                            self.tasks_cancelled.append(self.running_task)
                            self.broadcast_tasks_status('TASK_CANCEL', self.running_task)
                    except Exception as e:
                        if self.running_task:
                            self.running_task.logs.append("{}".format(ensure_unicode(e)))
                            self.running_task.logs.append(ensure_unicode(traceback.format_exc()))
                            self.running_task.summary = "{}".format(ensure_unicode(e))
                            self.tasks_error.append(self.running_task)
                            self.broadcast_tasks_status('TASK_ERROR', self.running_task)
                        self.logger.critical('Task error "%s": %s' % (self.running_task,ensure_unicode(e)))
                        try:
                            self.logger.debug(ensure_unicode(traceback.format_exc()))
                        except:
                            print("Traceback error")
                finally:
                    self.tasks_queue.task_done()
                    if self.running_task is not None:
                        try:
                            self.update_runstatus('')
                        except Exception:
                            self.logger.warning('Error reset runstatus : %s' % traceback.format_exc())
                    else:
                        # avoid looping too fast in case there are only tasks with start_not_before in future in queue
                        try:
                          self.check_scheduled_tasks()
                        except Exception:
                          self.logger.warning('Error checking scheduled tasks : %s' % traceback.format_exc())
                        if need_sleep:
                            time.sleep(5)

                    self.running_task = None
                    # trim history lists
                    if len(self.tasks_cancelled) > waptconfig.MAX_HISTORY:
                        del self.tasks_cancelled[:len(self.tasks_cancelled)-waptconfig.MAX_HISTORY]
                    if len(self.tasks_done) > waptconfig.MAX_HISTORY:
                        del self.tasks_done[:len(self.tasks_done)-waptconfig.MAX_HISTORY]
                    if len(self.tasks_error) > waptconfig.MAX_HISTORY:
                        del self.tasks_error[:len(self.tasks_error)-waptconfig.MAX_HISTORY]

            except queue.Empty:
                try:
                    self.update_runstatus('')
                except Exception:
                    self.logger.warning('Error reset runstatus : %s' % traceback.format_exc())

                try:
                    self.check_scheduled_tasks()
                except Exception:
                    self.logger.warning('Error checking scheduled tasks : %s' % traceback.format_exc())
                self.logger.debug("{} i'm still alive... but nothing to do".format(datetime.datetime.utcnow()))
                if need_sleep:
                    # avoid looping too fast in case there are only tasks with start_not_before in future in queue
                    time.sleep(10)

                gc.collect()
                global garbage_count
                global previous_garbage_count
                previous_garbage_count = copy.deepcopy(garbage_count)
                garbage_count = {}
                for c in gc.garbage:
                    if c.__class__ in garbage_count:
                        garbage_count[c.__class__] += 1
                    else:
                        garbage_count[c.__class__] = 1
                logger.debug(sorted(garbage_count.items(), key=lambda c: c[1], reverse=True))
                for k,c in garbage_count.items():
                    if c>previous_garbage_count.get(k,0):
                        logger.debug("%s: +%s (total:%s)" % (k,c - previous_garbage_count.get(k,0), c))
                logger.debug('gc count %s' % (gc.get_count(),))

            except Exception as e:
                if sys.platform == 'win32':
                    self.logger.critical('Unhandled error in task manager loop: %s. Sleeping 120s before restarting the service' % ensure_unicode(e))
                    time.sleep(120)
                    # ask waptsvc to restart service
                    self.logger.critical('Forced restart waptservice by waptsvc')
                    harakiri(10)
                else:  # TODO linux/macos : restart service
                    self.logger.critical('Unhandled error in task manager loop: %s. Exiting service' % ensure_unicode(e))
                    os._exit(1)

    def current_task_counter(self)->int:
        with self.status_lock:
            return self.tasks_counter

    def tasks_status(self)->Dict:
        """Returns list of pending, error, done tasks, and current running one"""
        with self.status_lock:
            return dict(
                running=self.running_task and self.running_task.as_dict(),
                pending=[task.as_dict() for task in sorted(self.tasks_queue.queue)],
                done=[task.as_dict() for task in self.tasks_done],
                cancelled=[task.as_dict() for task in self.tasks_cancelled],
                errors=[task.as_dict() for task in self.tasks_error],
                last_task_id=self.tasks_counter,
                last_event_id=self.events.last_event_id() if self.events else None,
            )

    def cancel_running_task(self) -> WaptTask:
        """Cancel running task. Returns cancelled task"""
        with self.status_lock:
            if self.running_task:
                try:
                    cancelled = self.running_task
                    self.tasks_error.append(self.running_task)
                    try:
                        self.running_task.kill()
                    except Exception as e:
                        print(e)
                finally:
                    self.running_task = None
                if cancelled:
                    self.tasks_cancelled.append(cancelled)
                    self.broadcast_tasks_status('TASK_CANCEL', cancelled)
                return cancelled
            else:
                return None

    def cancel_tasks(self, ids: List[int]) -> List[WaptTask]:
        """Cancel running or pending tasks with supplied ids.
        return cancelled tasks"""
        result = []
        with self.status_lock:
            if self.running_task and self.running_task.id is ids:
                result.append(self.running_task)
                try:
                    self.running_task.kill()
                    self.broadcast_tasks_status('TASK_CANCEL', self.running_task)
                except Exception as e:
                    print(e)
                finally:
                    self.running_task = None
            else:
                for task in self.tasks_queue.queue:
                    if task.id == id:
                        try:
                            result.append(task)
                            self.tasks_queue.queue.remove(task)
                            task.kill()
                            self.broadcast_tasks_status('TASK_CANCEL', task)
                        except Exception as e:
                            print(e)
            if result:
                self.tasks_cancelled.extend(result)
            return result

    def cancel_task(self, id) -> WaptTask:
        """Cancel running or pending task with supplied id.
            return cancelled task"""
        with self.status_lock:
            cancelled = None
            if self.running_task and self.running_task.id == id:
                cancelled = self.running_task
                try:
                    self.running_task.kill()
                except Exception as e:
                    print(e)
                finally:
                    self.running_task = None
            else:
                for task in self.tasks_queue.queue:
                    if task.id == id:
                        cancelled = task
                        self.tasks_queue.queue.remove(task)
                        break
                if cancelled:
                    try:
                        cancelled.kill()
                    except Exception as e:
                        print(e)
            if cancelled:
                self.tasks_cancelled.append(cancelled)
                self.broadcast_tasks_status('TASK_CANCEL', cancelled)
            return cancelled

    def cancel_all_tasks(self):
        """Cancel running and pending tasks. Returns list of cancelled tasks"""
        with self.status_lock:
            cancelled = []
            while not self.tasks_queue.empty():
                cancelled.append(self.tasks_queue.get())

            for task in cancelled:
                self.tasks_cancelled.append(task)
                self.broadcast_tasks_status('TASK_CANCEL', task)

            if self.running_task:
                try:
                    cancelled.append(self.running_task)
                    self.tasks_error.append(self.running_task)
                    try:
                        self.running_task.kill()
                    except Exception as e:
                        print(e)
                    self.broadcast_tasks_status('TASK_CANCEL', self.running_task)
                finally:
                    self.running_task = None
            return cancelled

    def start_ipaddr_monitoring(self):
        if sys.platform == 'win32':
            nac = ctypes.windll.iphlpapi.NotifyAddrChange

            def addr_change(taskman):
                while True:
                    nac(0, 0)
                    delayed_start = datetime2isodate(datetime.datetime.utcnow() + datetime.timedelta(seconds=20))
                    taskman.add_task(WaptNetworkReconfig(created_by='IPADDR_MONITORING',start_not_before=delayed_start))

            nm = threading.Thread(target=addr_change, args=(self,), name='ip_monitoring')
            nm.daemon = True
            nm.start()
            self.logger.debug("Wapt network address monitoring started")

    def start_network_monitoring(self):
        if sys.platform == 'win32':
            nrc = ctypes.windll.iphlpapi.NotifyRouteChange

            def connected_change(taskman):
                while True:
                    nrc(0, 0)
                    delayed_start = datetime2isodate(datetime.datetime.utcnow() + datetime.timedelta(seconds=20))
                    taskman.add_task(WaptNetworkReconfig(created_by='NETWORK_MONITORING',start_not_before=delayed_start))

            nm = threading.Thread(target=connected_change, args=(self,), name='network_monitoring')
            nm.daemon = True
            nm.start()
            self.logger.debug("Wapt connection monitor started")

    def running_and_pending_count(self)->int:
        res = self.tasks_queue.qsize()
        if self.running_task:
            res += 1
        return res

    def __str__(self):
        return "\n".join(self.tasks_status())


def install_service():
    """Setup waptservice as a windows Service managed by waptsvc
    >>> install_service()
    """
    if setuphelpers.service_installed('waptservice'):
        if not setuphelpers.service_is_stopped('waptservice'):
            logger.info('Stop running waptservice')
            setuphelpers.service_stop('waptservice')
            while not setuphelpers.service_is_stopped('waptservice'):
                logger.debug('Waiting for waptservice to terminate')
                time.sleep(2)
        logger.info('Unregister existing waptservice')
        setuphelpers.service_delete('waptservice')

    #service_binary = os.path.abspath(os.path.join(wapt_root_dir, 'waptsvc.exe'))
    # make a symlink for waptsvc.exe -> waptservice.exe
    service_link = os.path.abspath(os.path.join(wapt_root_dir, 'waptservice.exe'))
    #if os.path.exists(service_link):
    #    os.unlink(service_link)
    #os.symlink(service_binary,service_link)

    run('"%s" --install' % service_link)

    logger.info('Allow authenticated users to start/stop waptservice')
    sc_path = makepath(win32api.GetSystemDirectory(), 'sc.exe')
    if waptconfig.allow_user_service_restart:
        run('"%s" sdset waptservice D:(A;;CCLCSWRPWPDTLOCRRC;;;SY)(A;;CCDCLCSWRPWPDTLOCRSDRCWDWO;;;BA)(A;;CCLCSWLOCRRC;;;IU)(A;;CCLCSWLOCRRC;;;SU)(A;;CCLCSWRPWPDTLOCRRC;;;S-1-5-11)S:(AU;FA;CCDCLCSWRPWPDTLOCRSDRCWDWO;;;WD)' % sc_path)
    else:
        run('"%s" sdset waptservice D:(A;;CCLCSWRPLORC;;;AU)(A;;CCDCLCSWRPWPDTLOCRSDRCWDWO;;;BA)(A;;CCDCLCSWRPWPDTLOCRSDRCWDWO;;;SY)S:(AU;FA;CCDCLCSWRPWPDTLOSDRCWDWO;;;WD)' % sc_path)


def setup_logging(config=None):
    loglevel = config.loglevel
    for log in WAPTLOGGERS:
        sublogger = logging.getLogger(log)
        if sublogger:
            if hasattr(config, 'loglevel_%s' % log) and getattr(config, 'loglevel_%s' % log):
                setloglevel(sublogger, getattr(config, 'loglevel_%s' % log))
            else:
                if log == 'wapttasks':
                    setloglevel(sublogger, 'info')
                else:
                    setloglevel(sublogger, loglevel)
    if pywaptwua:
        if hasattr(config, 'loglevel_waptwua') and getattr(config, 'loglevel_waptwua'):
            pywaptwua.set_loglevel(getattr(config, 'loglevel_waptwua'))



def secure_wapt_db(waptconfig: WaptServiceConfig) -> bool:
    """move wapt private sqlite db in the private_dir location
    """
    old_db_path = makepath(waptconfig.wapt_base_dir,'db','waptdb.sqlite')
    new_db_path = makepath(waptconfig.private_dir,'db','waptdb.sqlite')
    if not os.path.isfile(new_db_path) and os.path.isfile(old_db_path):
        try:
            os.makedirs(os.path.dirname(new_db_path))
            os.rename(old_db_path, new_db_path)
            return True
        except:
            return False
    return True

if __name__ == "__main__":
    usage = """\
    %prog -c configfile [action]

    WAPT Service.

    action is either :
      <nothing> : run service in foreground
      install   : install as a Windows service managed by waptsvc

    """

    if sys.platform == 'win32':
        try:
            pythoncom.CoInitializeEx(pythoncom.COINIT_MULTITHREADED)
        except:
            pass # can not be initialized 2 times

    parser = OptionParser(usage=usage, version='service ' + __version__+' common.py '+common.__version__+' setuphelpers.py '+setuphelpers.__version__)
    parser.add_option("-c", "--config", dest="config", default=os.path.join(wapt_root_dir, 'wapt-get.ini'), help="Config file full path (default: %default)")
    parser.add_option("-l", "--loglevel", dest="loglevel", default=None, type='choice',  choices=['debug', 'warning', 'info', 'error', 'critical'], metavar='LOGLEVEL', help="Loglevel (default: warning)")
    parser.add_option("-d", "--devel", dest="devel", default=False, action='store_true', help="Enable debug mode (for development only)")
    parser.add_option("--waptbasedir", dest="wapt_base_dir", default=None, help="Force a different wapt-base-dir then default dir of waptutils.py. (default: %default)")
    # this is handled by wapt-get.exe waptservice python launcher
    parser.add_option("--peercache", dest="peercache", default=False,action='store_true', help="Enable peer cache udp listener and http server (default: %default)")

    for log in WAPTLOGGERS:
        parser.add_option('--loglevel_%s' % log, dest='loglevel_%s' % log, default=None, type='choice',
                          choices=['debug', 'warning', 'info', 'error', 'critical'],
                          metavar='LOGLEVEL', help='Loglevel %s (default: warning)' % log)

    (options, args) = parser.parse_args()

    if args and 'install' in args[:2]:
        print('Installing waptservice')
        install_service()
        print('Done')
        sys.exit(0)

    waptconfig.config_filename = options.config
    waptconfig.load(merge_config_packages=True, wapt_base_dir=options.wapt_base_dir)
    secure_wapt_db(waptconfig)

    for att in options.__dict__:
        if hasattr(waptconfig, att) and getattr(options, att) is not None:
            setattr(waptconfig, att, getattr(options, att))

    setup_logging(waptconfig)

    if waptconfig.log_to_windows_events:
        try:
            from logging.handlers import NTEventLogHandler
            hdlr = NTEventLogHandler('waptservice')
            logger.addHandler(hdlr)
        except Exception as e:
            logger.warning('Unable to initialize windows log Event handler: %s' % e)

    #for log in WAPTLOGGERS:
    for log in ['waptservice']:
        for f in glob.glob(os.path.join(wapt_root_dir,'log','%s-*.log' % log)):
            delta = datetime.date.today() - datetime.date.fromtimestamp(os.path.getmtime(f))
            if delta.days > int(waptconfig.log_retention_days):
                os.unlink(f)

    hdlr = logging.StreamHandler()
    hdlr.setFormatter(
        logging.Formatter('%(asctime)s [%(name)s %(threadName)s %(thread)s] %(levelname)s %(message)s'))
    rootlogger = logging.getLogger()
    rootlogger.addHandler(hdlr)
    setloglevel(rootlogger, options.loglevel)

    # app.logger.removeHandler(default_handler)
    # app.logger.addHandler(hdlr)

    # setup basic settings
    if sys.platform == 'win32':
        apply_host_settings(waptconfig)
        # waptwua
        wapt = Wapt(config_filename=waptconfig.config_filename)
        wapt.init_waptwua(options.loglevel_waptwua)


    tasks_logger.info('Service Configuration: %s' % waptconfig)

    # starts one WaptTasksManager
    tasks_logger.info('Starting task queue')
    task_manager = WaptTaskManager(config_filename=waptconfig.config_filename, wapt_base_dir = waptconfig.wapt_base_dir)

    task_manager.daemon = True
    task_manager.start()
    app.task_manager = task_manager
    if sys.platform == 'win32':
        waptwua_api.task_manager = task_manager

    tasks_logger.info('Tasks queue running')
    try:
        sio = WaptSocketIOClient(waptconfig.config_filename, task_manager=task_manager, wapt_base_dir=waptconfig.wapt_base_dir)
        sio_logger = logging.getLogger('socketIO-client-2')
        hdlr = logging.StreamHandler()
        hdlr.setFormatter(
            logging.Formatter('%(asctime)s [%(name)-15s] %(levelname)s %(message)s'))
        sio_logger.addHandler(hdlr)

        sio.start()
        if options.loglevel:
            setloglevel(sio_logger, options.loglevel)
        else:
            setloglevel(sio_logger, waptconfig.loglevel)

        if options.devel:
            tasks_logger.info('Starting local dev waptservice...')
            app.run(host='127.0.0.1', port=8088, debug=False)
        else:
            port_config = []
            if waptconfig.waptservice_port:
                tasks_logger.info('Starting waitress waptservice on port %s' % waptconfig.waptservice_port)
                waitress_logger = logging.getLogger('waitress')
                if options.loglevel:
                    setloglevel(waitress_logger, options.loglevel)
                else:
                    setloglevel(waitress_logger, waptconfig.loglevel)

                server = cheroot.wsgi.Server(("127.0.0.1",waptconfig.waptservice_port), app, numthreads=10,max=20, )
                if not waptconfig.disable_tls:
                    ssl_context = (makepath(waptconfig.public_dir,'localservice.crt'),makepath(waptconfig.private_dir,'localservice.pem'))
                    if not os.path.isdir(waptconfig.public_dir):
                        os.makedirs(waptconfig.public_dir)
                    ssl_local_key = SSLPrivateKey(ssl_context[1])
                    ssl_local_key.create()
                    ssl_local_key.save_as_pem()
                    ssl_local_cert = ssl_local_key.build_sign_certificate(cn='localhost', is_ca=True, is_server_auth=True, altnames=['127.0.0.1','localhost'], validity_days=3)
                    ssl_local_cert.save_as_pem(ssl_context[0])

                    ssl_adapter = BuiltinSSLAdapter(ssl_context[0], ssl_context[1], ciphers=SAFE_CIPHERS)
                    if sys.platform=='win32' and win32api.GetVersionEx()[0:2]<=(6,1):
                        ssl_adapter.context.minimum_version = ssl.TLSVersion.TLSv1_1
                    else:
                        ssl_adapter.context.minimum_version = ssl.TLSVersion.TLSv1_2
                    server.ssl_adapter = ssl_adapter

                server.start()
    finally:
        tasks_logger.info('Waptservice stopped')
