#!/usr/bin/env python3
##
## -----------------------------------------------------------------
##    This file is part of WAPT Software Deployment
##    Copyright (C) 2012 - 2024  Tranquil IT https://www.tranquil.it
##    All Rights Reserved.
##
##    WAPT helps systems administrators to efficiently deploy
##    setup, update and configure applications.
## ------------------------------------------------------------------
##
import pyldap

def check_credentials_ad(conf, username, password, list_group=[], use_service_account=False):
    """Authenticate against ldap with username and password

    Returns:
        list of groups
    """
    nesting_group_manage = conf.get("ldap_nesting_group_support",True)

    ldap_auth_server = conf.get('ldap_auth_server', '')
    dc_base_dn = conf.get('ldap_auth_base_dn')

    domain_name = conf.get('ad_domain_name')

    if not domain_name:
        domain_name = dc_base_dn.lower().split('dc=', 1)[-1].replace('dc=', '.').replace(',', '')

    auth_ok = {'success': False, 'groups': [], 'error': False, 'msg': ''}

    if not '@' in username:
        full_username = '%s@%s' % (username, domain_name)
    else:
        full_username = username

    if use_service_account:
        bind_username = conf['ldap_account_service_login']
        password = conf['ldap_account_service_password']
    else:
        bind_username = full_username

    try:
        client = pyldap.PyLdapClient(domain_name=domain_name,username=bind_username,password=password,ldap_server_name=ldap_auth_server)

        if str(ldap_auth_server).lower().startswith('ldaps:'):
            if not conf['verify_cert_ldap']:
                client.tls_context.ignore_certificate_errors = True
            r = client.bind()
            if type(r) == bool:
                r=[r,'']
        else:
            r = client.bind_sasl_kerberos()

        if not r[0]:
            auth_ok['success'] =False
            auth_ok['msg'] = str(client.result_string())
        else:
            auth_ok['success'] =True
            auth_ok['groups'] = client.get_authorized_user_groups(user_cn=username, check_for_groups=list_group, user_base_dn=dc_base_dn, group_base_dn=dc_base_dn, group_nested=nesting_group_manage)
            if not auth_ok['groups']:
                auth_ok['groups'] = []

    except Exception as e:
        auth_ok['success'] = False
        auth_ok['groups'] = []
        auth_ok['error'] = True
        auth_ok['msg'] = str(e)
    finally:
        return auth_ok


