#!/usr/bin/env python3
##
## -----------------------------------------------------------------
##    This file is part of WAPT Software Deployment
##    Copyright (C) 2012 - 2024  Tranquil IT https://www.tranquil.it
##    All Rights Reserved.
##
##    WAPT helps systems administrators to efficiently deploy
##    setup, update and configure applications.
## ------------------------------------------------------------------
##

import time
import sys
import os
import logging
import threading
import platform
import requests
import random
import datetime
import traceback

try:
    wapt_root_dir = os.path.abspath(os.path.join(os.path.dirname(__file__), '..'))
except:
    wapt_root_dir = 'c:/tranquilit/wapt'

#from waptutils import __version__

import socketio

import setuphelpers

# wapt specific stuff
from waptutils import is_between_two_times
from waptutils import default_http_headers
from waptutils import harakiri, datetime2isodate
from waptutils import ensure_list
from waptpackage import EWaptException
from waptcrypto import SSLVerifyException
from urllib3.exceptions import InsecureRequestWarning

from common import Wapt
import waptcrypto

from waptservice.waptservice_common import waptservice_remote_actions, waptconfig, WaptServiceConfig
from waptservice.waptservice_common import WaptUpdate, WaptUpgrade, WaptUpdateServerStatus, WaptRegisterComputer, WaptDownloadUpgrade
from waptservice.waptservice_common import WaptCleanup, WaptPackageInstall, WaptPackageRemove, WaptPackageForget, WaptLongTask, WaptAuditPackage
from waptservice.waptservice_common import _

from waptservice.repositories import WaptSyncRepo
from waptservice.enterprise import WaptRunSessionSetup

try:
    import waptlicences
except ImportError:
    waptlicences = None

if sys.platform == 'win32':
    import pythoncom

logger = logging.getLogger('waptws')
tasks_logger = logging.getLogger('wapttasks')

# Websocket stuff
##### API V2 #####


def make_response(result={}, success=True, error_code='', msg='', uuid=None, request_time=None):
    data = dict(
        uuid=uuid,
        success=success,
        msg=msg,
        error_code=error_code,
        result=result,
        request_time=request_time,
    )
    return data


def make_response_from_exception(exception, error_code='', uuid=None, request_time=None):
    """Create a standard answer for websocket callback exception

    Returns:
        dict: {success : False, msg : message from exception, error_code : classname of exception if not provided}
   """
    if not error_code:
        error_code = type(exception).__name__.lower()

    data = dict(
        uuid=uuid,
        success=False,
        msg="Error on client: %s" % repr(exception),
        error_code=error_code,
        request_time=request_time,
    )
    return data


def wait_for_event_send_tasks(task_manager, last_received_event_id, timeout):
    data = None
    start_time = time.time()
    while True:
        actual_last_event_id = task_manager.events.last_event_id()
        if actual_last_event_id is not None and actual_last_event_id <= last_received_event_id:
            data = {'last_event_id': task_manager.events.last_event_id()}
            if time.time() - start_time > timeout:
                break
        elif actual_last_event_id is None or actual_last_event_id > last_received_event_id:
            data = task_manager.tasks_status()
            break

        if time.time() - start_time > timeout:
            break

        # avoid eating cpu
        time.sleep(0.1)

    # send back the tasks to the console
    return data

class WaptSocketIORemoteCalls(socketio.ClientNamespace):

    _wapt = None
    host_uuid = None
    task_manager = None
    config_filename = None
    config = None

    @property
    def wapt(self)->Wapt:
        # thread local wapt...
        if not hasattr(self,'_wapt') or self._wapt is None:
            self._wapt = Wapt(config_filename = self.config_filename, wapt_base_dir=self.config.wapt_base_dir)
            tasks_logger.info('Socketio new Wapt intance has Thread ID: %s' % threading.get_ident())

        return self._wapt

    def check_action_signature(self,action,required_attributes):
        return waptcrypto.check_action_signature(self.wapt.cabundle, action, required_attributes, waptconfig.signature_clockskew)

    def on_trigger_host_action(self, args):
        try:
            start_time = time.time()
            actions = args
            if not isinstance(actions, list):
                actions = [actions]

            for action in actions:
                required_attributes = ['uuid', 'action']
                tasks_logger.info('Host actions "%s" triggered by SocketIO by %s' % (action['action'], action['signer']))
                name = action['action']
                uuid = action['uuid']
                if uuid != self.host_uuid:
                    raise Exception(_('Task is not targeted to this host. task''s uuid does not match host''uuid'))
                if name in ['trigger_install_packages', 'trigger_remove_packages', 'trigger_forget_packages']:
                    required_attributes.append('packages')
                if name in ['trigger_change_description']:
                    required_attributes.append('computer_description')
                if name in waptservice_remote_actions:
                    required_attributes.extend(waptservice_remote_actions[name].required_attributes)
                verified_by = self.check_action_signature(action, required_attributes=required_attributes)
                if not verified_by:
                    raise SSLVerifyException(_('Bad signature for action %s, aborting') % name)

            result = []
            for action in actions:
                name = action['action']
                if name in ['trigger_cancel_all_tasks']:
                    data = [t.as_dict() for t in self.task_manager.cancel_all_tasks()]
                    result.append(data)

                elif name in ['trigger_host_register', 'trigger_host_update_server_status']:
                    if name == 'trigger_host_update_server_status':
                        task = WaptUpdateServerStatus(
                          force=True)
                    elif name == 'trigger_host_register':
                        task = WaptRegisterComputer(computer_description=action.get('computer_description', None))
                        task.force = action.get('force', False)
                    task.notify_user = action.get('notify_user', False)
                    task.notify_server_on_finish = action.get('notify_server', False)
                    task.created_by = verified_by
                    tasks_logger.info('task "%s" to %s triggered by %s' % (task,name,verified_by))
                    data = self.task_manager.add_task(task).as_dict()
                    result.append(data)

                elif name in ['trigger_change_description']:
                    desc = action.get('computer_description', None)
                    tasks_logger.info('task "Change host description" to %s triggered by %s' % (desc,verified_by))
                    if desc is not None:
                        self.wapt.write_param('host_description',desc)
                        try:
                            setuphelpers.set_computer_description(desc)
                        except Exception as e:
                            logger.info('Unable to change system computer description to %s: %s' % (desc, e))

                        msg = _('Computer description of %s changed to %s') % (setuphelpers.get_hostname(), desc)
                        result.append(dict(success=True,
                                           msg=msg,
                                           result=msg,
                                           ))
                        if action.get('notify_server', False):
                            task = WaptUpdate(created_by=verified_by)
                            task.notify_server_on_finish = True
                            self.task_manager.add_task(task)
                            result.append(task.as_dict())

                elif name == 'trigger_host_update':
                    tasks_logger.info('task "Host update" triggered remotely by %s' % (verified_by))
                    notify_user = action.get('notify_user', False)
                    notify_server_on_finish = action.get('notify_server', False)
                    force = action.get('force', False)
                    download_packages = action.get('download', False)

                    result.append(self.task_manager.add_task(
                        WaptUpdate(
                            created_by=verified_by,
                            force=force,
                            notify_user=notify_user,
                            )).as_dict())
                    if download_packages:
                        result.append(self.task_manager.add_task(
                            WaptDownloadUpgrade(
                                created_by=verified_by,
                                force=force,
                                notify_user=notify_user,
                                from_websocket=True
                                )).as_dict())

                elif name == 'trigger_host_upgrade':
                    tasks_logger.info('task "Host upgrade" triggered remotely by %s' % (verified_by))
                    notify_user = action.get('notify_user', False)
                    notify_server_on_finish = action.get('notify_server', False)
                    force = action.get('force', False)
                    only_priorities = action.get('only_priorities', None)
                    only_if_not_process_running = action.get('only_if_not_process_running', False)
                    update_packages = action.get('update', True)

                    if update_packages:
                        result.append(self.task_manager.add_task(WaptUpdate(created_by=verified_by,force=force, notify_user=notify_user, notify_server_on_finish=False,
                                                                            )).as_dict())

                    result.append(self.task_manager.add_task(WaptDownloadUpgrade(created_by=verified_by,
                                                                                 force=force,
                                                                                 notify_user=notify_user,
                                                                                 from_websocket=True
                                                                                )).as_dict())

                    result.append(self.task_manager.add_task(WaptUpgrade(notify_user=notify_user,
                                                                         created_by=verified_by,
                                                                         notify_server_on_finish=notify_server_on_finish,
                                                                         only_priorities=only_priorities,
                                                                         only_if_not_process_running=only_if_not_process_running,
                                                                         force=force
                                                                         )).as_dict())

                    result.append(self.task_manager.add_task(WaptCleanup(notify_user=False, created_by=verified_by, priority=200)).as_dict())
                    if self.wapt.is_enterprise() and setuphelpers.get_loggedinusers():
                        result.append(self.task_manager.add_task(WaptRunSessionSetup(created_by=verified_by)).as_dict())

                elif name in ['trigger_install_packages', 'trigger_remove_packages', 'trigger_forget_packages']:
                    packagenames = action['packages']
                    only_priorities = action.get('only_priorities', None)
                    only_if_not_process_running = action.get('only_if_not_process_running', False)

                    for packagename in packagenames:
                        if name == 'trigger_install_packages':
                            task = WaptPackageInstall(packagenames=packagename)
                        elif name == 'trigger_remove_packages':
                            task = WaptPackageRemove(packagenames=packagename)
                        elif name == 'trigger_forget_packages':
                            task = WaptPackageForget(packagenames=packagename)
                        task.force = action.get('force', False)
                        task.notify_user = action.get('notify_user', False)
                        task.notify_server_on_finish = action.get('notify_server', False)
                        task.created_by = verified_by
                        task.only_priorities = only_priorities
                        task.only_if_not_process_running = only_if_not_process_running

                        tasks_logger.info('Task %s triggered remotely by %s' % (task,verified_by))
                        result.append(self.task_manager.add_task(task).as_dict())

                    if name == 'trigger_install_packages' and self.wapt.is_enterprise():
                        self.task_manager.add_task(WaptAuditPackage(packagenames=packagenames,
                                                                    created_by=verified_by,
                                                                    force=task.force,
                                                                    notify_user=task.notify_user,
                                                                    notify_server_on_finish=task.notify_server_on_finish,
                                                                    priority=200)).as_dict()
                        if setuphelpers.get_loggedinusers():
                            self.task_manager.add_task(WaptRunSessionSetup(created_by=verified_by))


                elif name == 'trigger_waptservicerestart':
                    tasks_logger.info('Waptservice restart triggered by %s' % verified_by)

                    if platform.system() == 'Windows':
                        harakiri(10)
                    elif platform.system() == 'Darwin':
                        msg = setuphelpers.run("launchctl kickstart -k system/it.tranquil.waptservice")
                    elif platform.system() == 'Linux':
                        msg = setuphelpers.run('systemctl restart waptservice')
                    else:
                        msg = 'Restart not supported'
                        tasks_logger.critical(msg)
                    result.append(dict(success=True, msg=msg, result=msg))

                elif name == 'unregister_computer':
                    self.wapt.delete_param('last_update_server_hashes')
                    self.wapt.delete_param('last_audit_data_server_date')
                    self.wapt.delete_param('server_uuid')
                    if os.path.isfile(self.wapt.get_host_certificate_filename()):
                        os.unlink(self.wapt.get_host_certificate_filename())
                    self.disconnect()

                elif name == 'trigger_longtask':
                    task = WaptLongTask()
                    task.force = args.get('force', False)
                    task.notify_user = args.get('notify_user', False)
                    task.notify_server_on_finish = args.get('notify_server', False)
                    task.created_by = verified_by
                    result.append(self.task_manager.add_task(task).as_dict())
                elif name in waptservice_remote_actions:
                    waptservice_remote_actions[name].trigger_action(self, action, verified_by)
                else:
                    raise EWaptException('Unhandled remote action %s' % name)

            # self.emit('trigger_update_result',{'result':data})
            #if result_callback:
            return make_response(result, uuid=self.host_uuid, request_time=time.time()-start_time)
        except BaseException as e:
            logger.warning('Exception for actions %s: %s' % (repr(args), repr(e)))
            #if result_callback:
            #    result_callback(make_response_from_exception(e, uuid=self.wapt.host_uuid, request_time=time.time()-start_time))

    def on_get_tasks_status(self, args):
        try:
            self.check_action_signature(args,['uuid'])

            uuid = args.get('uuid', '')
            if uuid != self.host_uuid:
                raise Exception(_('Task is not targeted to this host. task''s uuid does not match host''uuid'))

            timeout = float(args.get('timeout', '10.0'))
            last_received_event_id = int(args.get('last_event_id', '-1'))

            data = {}
            if self.task_manager.events:
                data = wait_for_event_send_tasks(self.task_manager, last_received_event_id, timeout)
            else:
                time.sleep(0.5)

            return make_response(result = data, uuid=self.host_uuid)

        except BaseException as e:
            logger.info('Exception for actions %s: %s' % (repr(args), repr(e)))
            return make_response_from_exception(e, uuid=self.host_uuid)

    def on_wapt_ping(self, args=None):
        logger.debug('wapt_ping... %s' % (args,))
        self.emit('wapt_pong')

    def on_sync_remote_repo(self, args=None):
        tasks_logger.info('Synchronize local remote repo %s' % (args,))
        if waptconfig.enable_remote_repo and self.wapt.is_enterprise() and not(waptconfig.sync_only_forced):
            if not(waptconfig.local_repo_time_for_sync_start) or is_between_two_times(waptconfig.local_repo_time_for_sync_start, waptconfig.local_repo_time_for_sync_end):
                try:
                    self.task_manager.add_task(WaptSyncRepo(notify_user=False, created_by='SERVER', socketio_client=self))
                except Exception as e:
                    logger.debug('Error syncing local repo with server repo : %s' % e)
            else:
                self.emit("synchronization_not_in_time_range")
        else:
            self.emit("synchronization_not_a_local_remote_repo")

    def on_sync_remote_repo_force(self, args=None):
        tasks_logger.info('Synchronize local remote repo %s' % (args,))
        if waptconfig.enable_remote_repo and self.wapt.is_enterprise():
            try:
                self.task_manager.add_task(WaptSyncRepo(notify_user=False, created_by='SERVER', socketio_client=self))
            except Exception as e:
                logger.debug('Error syncing local repo with server repo : %s' % e)
        else:
            self.emit("synchronization_not_a_local_remote_repo")

    def on_audit_remote_repo(self, args=None):
        tasks_logger.info('Auditing files of remote repo %s' % (args,))
        if waptconfig.enable_remote_repo and self.wapt.is_enterprise():
            try:
                self.task_manager.add_task(WaptSyncRepo(notify_user=False, created_by='SERVER', audit=True, socketio_client=self))
            except Exception as e:
                logger.debug('Error auditing local repo with server repo : %s' % e)
        else:
            self.emit("synchronization_not_a_local_remote_repo")

    def on_wapt_force_reconnect(self, args=None):
        tasks_logger.info('Force disconnect from server...')
        self.disconnect()

    def on_message(self, message=None):
        logger.debug('socket.io message : %s' % message)

    def on_event(self, event, *args):
        logger.debug('socket.io event : %s, args: %s' % (event, args))

    def on_connect(self):
        logger.info('socket.io CONNECT')
        self.wapt.check_peercache_status()

        self.enqueue_audit(self._wapt, trigger='on_connect', notify_server_on_finish = not self.config.update_server_status_on_connect)
        if self.config.update_packages_on_connect:
            self.enqueue_update_packages(self._wapt)
        elif self.config.update_server_status_on_connect:
            self.enqueue_update_server_status(self._wapt)

    def on_disconnect(self):
        logger.info('socket.io DISCONNECT')
        self.enqueue_audit(self._wapt, trigger='on_disconnect',notify_server_on_finish=False)

    def enqueue_update_packages(self,wapt) -> dict:
        task = WaptUpdate()
        task.notify_user = False
        task.notify_server_on_finish = self.config.update_server_status_on_connect
        task.created_by = 'RECONNECT'
        task.start_not_before = datetime2isodate(datetime.datetime.utcnow() + datetime.timedelta(seconds=12 + random.random()*20)) # 2 + 20 seconds to spread updates on server for I/O
        tasks_logger.info('Enqueued update packages')
        return self.task_manager.add_task(task).as_dict()

    def enqueue_update_server_status(self,wapt) -> dict:
        task = WaptUpdateServerStatus()
        task.notify_user = False
        task.notify_server_on_finish = False
        task.created_by = 'RECONNECT'
        task.start_not_before = datetime2isodate(datetime.datetime.utcnow() + datetime.timedelta(seconds=12 + random.random()*20)) # 2 + 20 seconds to spread updates on server for I/O
        tasks_logger.info('Enqueued update_server_status')
        return self.task_manager.add_task(task).as_dict()

    def enqueue_audit(self,wapt: Wapt, trigger='on_connect',notify_server_on_finish=False) -> dict:
        packages = list(wapt.waptdb.query("""select package_uuid,package,audit_schedule from wapt_localstatus where install_status='OK' and audit_schedule like '%"""+trigger+"""%'"""))
        package_names = [p['package'] for p in packages if trigger in ensure_list(p['audit_schedule'] or [])]
        if package_names:
            task = WaptAuditPackage(packagenames=package_names)
            task.notify_user = False
            task.notify_server_on_finish = notify_server_on_finish
            task.force = True
            task.created_by = trigger.upper()
            task.start_not_before = datetime2isodate(datetime.datetime.utcnow() + datetime.timedelta(seconds=2 + random.random()*10)) # 2 + 10 seconds to spread updates on server for I/O
            tasks_logger.info('Enqueued audit for %s' % (task.packagenames,))
            return self.task_manager.add_task(task).as_dict()



def get_requests_socketclient_session(url=None, cert=None, verify=True, proxies={'http': None, 'https': None}, **kwargs):
    """Returns a requests Session which is aware of client cert auth with password protected key
    Disable use of environ.

    Args:
        url (str): base prefix url for which the session is created
        cert (tuple) : (certfilename,pem encoded key filename, key password)
        verify (bool or str) : verify server certificate. Id str, path to trusted CA bundle

    Returns:
        Session
    """
    result = requests.Session()
    # be sure to not use HTTP_PROXY or HTTPS_PROXY environ variable
    result.trust_env = False
    result.headers = default_http_headers()
    result.verify = verify
    result.proxies = proxies
    if not verify:
        requests.packages.urllib3.disable_warnings(InsecureRequestWarning)  # pylint: disable=no-member

    result.cert = cert
    return result


class WaptSocketIOClient(threading.Thread):
    def __init__(self, config_filename='c:/wapt/wapt-get.ini', task_manager=None, request_timeout=None, wapt_base_dir=None):
        threading.Thread.__init__(self)
        self.name = 'SocketIOClient'
        self.config_filename = config_filename
        self.task_manager = task_manager
        self.config = WaptServiceConfig(config_filename)
        if wapt_base_dir is not None:
            self.config.wapt_base_dir=wapt_base_dir
        self.socketio_client = None
        self.server_authorization_token = None
        self.connect_params = {}
        if request_timeout is None:
            request_timeout = self.config.websockets_request_timeout
        self.request_timeout = request_timeout


    def enqueue_register_computer(self,wapt):
        task = WaptRegisterComputer()
        task.notify_user = False
        task.notify_server_on_finish = False
        task.created_by = 'AUTO-REGISTER'
        tasks_logger.info('Enqueued register_computer')
        return self.task_manager.add_task(task).as_dict()

    def websocket_url(self):
        return "%s://%s:%s" % (self.config.websockets_proto, self.config.websockets_host,self.config.websockets_port)

    def run(self):
        if sys.platform == 'win32':
            pythoncom.CoInitializeEx(pythoncom.COINIT_MULTITHREADED)

        self.config.load()

        reconnect_delay = random.randint(2,50) / 10.0

        while True:
            try:
                tmp_wapt = Wapt(config_filename=self.config.config_filename, wapt_base_dir=self.config.wapt_base_dir)
                tasks_logger.info('Starting socketio on "%s://%s:%s" Thread id %s' % (self.config.websockets_proto, self.config.websockets_host, self.config.websockets_port, threading.current_thread().ident))
                logger.info('Certificate checking : %s' % self.config.websockets_verify_cert)

                def update_connect_params(wapt):
                    config_updated = wapt.reload_config_if_updated()
                    if config_updated:
                        self.config.load()
                    if not self.config.websockets_host:
                        return False
                    if (self.socketio_client and not self.socketio_client.connected) or config_updated or not self.connect_params or self.connect_params.get('uuid') != wapt.host_uuid:
                        self.connect_params['uuid'] = wapt.host_uuid
                        return True
                    return False

                while True:
                    try:
                        need_reconnect = update_connect_params(tmp_wapt)
                        if need_reconnect:
                            tasks_logger.info('Socketio connection params have changed. Socketio needs reconnect')

                        if not self.socketio_client and self.config.websockets_host:
                            if self.config.waptserver.proxies and self.config.waptserver.proxies.get(self.config.websockets_proto, None) is not None:
                                proxies = self.config.waptserver.proxies
                                websocket_extra_options = {}
                            else:
                                # trick to workaround a bug in websocket module which tries to get proxy from environment
                                # we specify a not null proxy but deny use for all hosts (*)
                                proxies = None
                                websocket_extra_options = {'http_proxy_host':'localhost','http_proxy_port': 0, 'http_no_proxy': ['*']}


                            cert = (tmp_wapt.get_host_certificate_filename(), tmp_wapt.get_host_key_filename())

                            tasks_logger.info('Creating socketio client: %s client auth cert: %s proxies: %s verify_cert: %s' % (self.websocket_url(),cert,proxies,self.config.websockets_verify_cert))

                            self.socketio_client = socketio.Client(
                                reconnection = self.config.websockets_low_level_reconnect,
                                reconnection_attempts = 5,
                                reconnection_delay_max = self.config.websockets_retry_delay - 2,
                                logger = logger,
                                engineio_logger = logger,
                                request_timeout=self.request_timeout,
                                websocket_extra_options=websocket_extra_options,

                                ssl_verify=self.config.websockets_verify_cert,
                                http_session = get_requests_socketclient_session(url=self.websocket_url(),
                                    cert=cert,
                                    proxies=proxies,
                                    verify=self.config.websockets_verify_cert
                                    ),
                                )

                        if self.socketio_client and self.config.websockets_host:
                            if self.socketio_client.connected and need_reconnect:
                                tasks_logger.info('Disconnecting Socketio')
                                self.socketio_client.disconnect()
                                self.socketio_client.wait()

                                #start = time.time()
                                #while (self.socketio_client.eio.read_loop_task.isAlive() or self.socketio_client.eio.write_loop_task.isAlive()) and (time.time() - start < 10):
                                #    time.sleep(1)

                            if not self.socketio_client.connected:
                                if '/' in self.socketio_client.namespace_handlers:
                                    name_space = self.socketio_client.namespace_handlers['/']
                                else:
                                    name_space = WaptSocketIORemoteCalls()
                                    self.socketio_client.register_namespace(name_space)
                                name_space.config_filename = self.config_filename
                                name_space.task_manager = self.task_manager
                                name_space.config = self.config
                                name_space._wapt = tmp_wapt
                                name_space.host_uuid = tmp_wapt.host_uuid

                                tasks_logger.info('Connecting Socketio to %s' % (self.websocket_url(),))
                                self.socketio_client.connect(
                                    url=self.websocket_url(),
                                    headers = self.connect_params,
                                    socketio_path = self.config.websockets_root,
                                    transports = ['websocket','polling'])

                            if self.socketio_client.connected:
                                # reset reconnect_delay to a small time to speed up reconnection
                                reconnect_delay = random.randint(0,50) / 10.0
                                if need_reconnect:
                                    tasks_logger.info('WS read loop for %ss' % (self.config.websockets_check_config_interval,))

                                self.socketio_client.sleep(1)  # give the reconnect task time to start up
                                if self.socketio_client.eio.read_loop_task:
                                    # read messages for websockets_check_config_interval seconds.
                                    # when elapsed, we will check config.
                                    self.socketio_client.eio.read_loop_task.join(self.config.websockets_check_config_interval)
                                else:
                                    # we should retry later.
                                    tasks_logger.warning('WS EIO read loop failed')
                                    self.socketio_client.sleep(10)

                        else:
                            time.sleep(self.config.websockets_retry_delay)

                    except Exception as e:
                        if reconnect_delay <= 2.0:
                            reconnect_delay = 2.0
                        logger.warning('Exception %s, waiting %ss before retrying' %
                                    (repr(e), reconnect_delay))
                        #logger.debug(traceback.format_exc())
                        self.socketio_client = None
                        self.connect_params = {}
                        #if tmp_wapt.waptserver.available():
                        #    self.enqueue_register_computer(tmp_wapt)
                        time.sleep(reconnect_delay)
                        # increase
                        reconnect_delay = reconnect_delay + reconnect_delay * 3/4
                        if reconnect_delay > self.config.websockets_retry_delay:
                            reconnect_delay = self.config.websockets_retry_delay

            except Exception as e:
                logger.critical("Socketio client: Error loading Wapt: %s" % e)
                time.sleep(self.config.websockets_retry_delay)



if __name__ == "__main__":
    pass
