#!/usr/bin/env python3
##
## -----------------------------------------------------------------
##    This file is part of WAPT Software Deployment
##    Copyright (C) 2012 - 2024  Tranquil IT https://www.tranquil.it
##    All Rights Reserved.
##
##    WAPT helps systems administrators to efficiently deploy
##    setup, update and configure applications.
## ------------------------------------------------------------------
##
import functools
from common import Wapt
from flask import g, Response
from waptservice.waptservice_common import waptconfig


def wapt():
    """Flask request contextual cached Wapt instance access"""
    if not hasattr(g, 'wapt'):
        g.wapt = Wapt(config_filename=waptconfig.config_filename)
    # apply settings if changed at each wapt access...
    else:
        g.wapt.reload_config_if_updated()
    return g.wapt


def forbidden_no_valid_licence():
    """Sends a 403 response that tells there's no valid enterprise licence"""
    return Response(
        'Forbidden due to no valid enterprise licence.\n',
        403)


def requires_enterprise(f):
    """Flask route decorator which requires wapt.is_enterprise() to be True.
    If not return a 404 http status.

    Args:
        methods (list of str): list of authentication methods to check in turn
        require_any_of (None or list of str): requires that user has at least of these
                                                acls in any of its perimeters.

    Returns:
        decorator

    """
    @functools.wraps(f)
    def decorated(*args, **kwargs):
        if not wapt().is_enterprise():
            return forbidden_no_valid_licence()

        result = f(*args, **kwargs)
        return result
    return decorated
