# -*- coding: utf-8 -*-
from setuphelpers import *
import os
import shutil


app_name = "TeamViewer Portable"
zip_name = "TeamViewerPortable.zip"
app_dir = makepath(programfiles32, app_name)
app_path = makepath(app_dir, "TeamViewer.exe")
icon_path = app_path
audit_version_number = False
win64app = None


def get_installed_version(uninstallkey):
    for soft in installed_softwares(uninstallkey):
        return soft.get("version", None)
    return None


def install():
    # Declaring local variables
    package_version = control.get_software_version()
    installed_version = get_installed_version(app_name)
    ini_file = makepath(app_dir, "TeamViewer.ini")

    # Installing software
    if installed_version is None or Version(installed_version) < Version(package_version) or force:
        print("Installing: %s (%s)" % (app_name, control.get_software_version()))
        killalltasks(ensure_list(control.impacted_process))
        if isdir(app_dir) and force:
            remove_tree(app_dir)
        mkdirs(app_dir)
        print("Extracting: %s to: %s" % (zip_name, app_dir))
        unzipped_dir = zip_name.rsplit(".", 1)[0]
        # unzip(zip_name, app_dir)
        unzip(zip_name, unzipped_dir)
        copytree2(unzipped_dir, app_dir, shutil.ignore_patterns("TeamViewer.ini"), onreplace=default_overwrite)

        # Adding software to regitry
        print("Registering: %s to Windows Registry" % app_name)
        register_windows_uninstall(control, win64app=win64app)
        register_uninstall(app_name, win64app=win64app, install_location=app_dir, icon=icon_path)
    else:
        print("%s is already installed and up-to-date (%s)" % (app_name, control.get_software_version()))

    # Modifying ini file
    inifile_writestring(ini_file, "Settings", "nosave", "1")  # Avoid UAC
    inifile_writestring(ini_file, "Settings", "TermsOfUseAccepted", "1")  # Accept EULA and CLUF
    inifile_writestring(ini_file, "Settings", "UseNewUI", "0")  # Keep the old UI

    # Creating custom shortcuts
    create_desktop_shortcut(app_name, app_path)
    create_programs_menu_shortcut(app_name, app_path)

    # Adding Firewall Rules
    remove_netfirewallrule(app_name)
    add_netfirewallrule(app_name, app_path, app_name, profile="Domain,Private", protocol="UDP")
    add_netfirewallrule(app_name, app_path, app_name, profile="Domain,Private", protocol="TCP")


def audit():
    # Declaring local variables
    package_version = control.get_software_version()
    installed_version = get_installed_version(app_name)

    # Auditing software
    if installed_version is None:
        print("%s is not installed" % app_name)
        return "ERROR"
    elif Version(installed_version) != Version(package_version) and audit_version_number:
        print("%s is installed in version (%s) instead of (%s)" % (app_name, installed_version, package_version))
        return "WARNING"
    else:
        print("%s (%s) is installed" % (app_name, installed_version))
        return "OK"


def uninstall():
    # Uninstalling software
    killalltasks(ensure_list(control.impacted_process))
    if isdir(app_dir):
        remove_tree(app_dir)
    unregister_uninstall(app_name, win64app=win64app)

    # Removing shortcuts
    remove_desktop_shortcut(app_name)
    remove_programs_menu_shortcut(app_name)

    # Removing Firewall Rules
    remove_netfirewallrule(app_name)


def add_netfirewallrule(
    rule_name,
    app_path,
    group=None,
    direction="Inbound",
    profile=None,
    enabled=True,
    action="Allow",
    protocol=None,
):
    """
    Add a Windows Firewall rule using New-NetFirewallRule cmdlet in PowerShell for Windows 10 and newer,
    or netsh advfirewall for older Windows versions.

    https://learn.microsoft.com/troubleshoot/windows-server/networking/netsh-advfirewall-firewall-control-firewall-behavior
    https://learn.microsoft.com/powershell/module/netsecurity/new-netfirewallrule

    Args:
        rule_name (str): Display name for the firewall rule.
        app_path (str): Path to the program for which the rule is being created.
        group (str, optional): Group name for the firewall rule (only works with PowerShell).
        direction (str): Direction of the rule (Inbound or Outbound). Default: Inbound.
        profile (str or list, optional): Profile(s) to which the rule should apply (e.g., "Domain,Private"). Default: Any.
        enabled (bool): Specify if the created rule's state is Enabled (True) or not (False). Default: True.
        action (str): Action for the rule (Allow or Block). Default: Allow.
        protocol (str, optional): Protocol by name or number (e.g., "TCP", "UDP", "ICMPv4", or "ICMPv6"). Default: Any.

    Returns:
        waptutils.RunOutput: The result of the command execution.

    .. versionadded:: 2.5

    """
    if not isinstance(profile, list):
        profile = ensure_list(profile)
    profile = ",".join(profile)

    message = f"Adding Firewall Rule: {rule_name}"
    if direction:
        message += f" ({direction})"
    if protocol:
        message += f" ({protocol})"
    if action:
        message += f" ({action})"
    print(message)
    if windows_version() < WindowsVersions.Windows10:
        direction = "out" if direction.lower() == "Outbound".lower() else "in"
        enabled = "no" if not enabled else "yes"
        cmd_command = f'netsh advfirewall firewall add rule name="{rule_name}" dir={direction} action={action} program="{app_path}" enable={enabled}'
        if profile:  # any and all are working
            cmd_command += f' profile="{profile}"'
        if protocol:
            cmd_command += f' protocol="{protocol}"'
        result = run_notfatal(cmd_command)
    else:
        pwsh_command = (
            f'New-NetFirewallRule -DisplayName "{rule_name}" -Direction {direction} -Action {action} -Program "{app_path}" -Enabled {str(enabled)}'
        )
        if group:
            pwsh_command += f' -Group "{group}"'
        if profile:
            pwsh_command += f' -Profile "{profile}"'
        if protocol:
            pwsh_command += f' -Protocol "{protocol}"'
        result = run_powershell(pwsh_command, output_format="text")

    return result


def remove_netfirewallrule(rule_name):
    """
    Remove Windows Firewall rule using New-NetFirewallRule cmdlet in PowerShell for Windows 10 and newer,
    or netsh advfirewall for older Windows versions.

    Args:
        rule_name (str): Display name of the firewall rule to remove.

    Returns:
        waptutils.RunOutput: The result of the command execution.

    .. versionadded:: 2.5

    """
    print(f"Removing Firewall Rule: {rule_name}")
    if windows_version() < WindowsVersions.Windows10:
        result = run_notfatal(f'netsh advfirewall firewall delete rule name="{rule_name}"')
    else:
        result = run_powershell(
            f'Remove-NetFirewallRule -DisplayName "{rule_name}" -ErrorAction SilentlyContinue', output_format="text", accept_returncodes=[0, 1, 3010]
        )

    return result
