# -*- coding: utf-8 -*-
from setuphelpers import *
import tarfile
from typing import List, Optional

"""
sudo ./install.sh --help

Usage: ./install.sh [options]

Options:

    --uninstall                      only uninstall from the installation prefix
    --destdir=[<none>]               set installation root
    --prefix=[/usr/local]            set installation prefix
    --without=[<none>]               comma-separated list of components to not install
    --components=[<none>]            comma-separated list of components to install
    --list-components                list available components
    --sysconfdir=[/etc]              install system configuration files
    --bindir=[/bin]                  install binaries
    --libdir=[/lib]                  install libraries
    --datadir=[/share]               install data
    --mandir=[/share/man]            install man pages in PATH
    --docdir=[\<default\>]           install documentation in PATH
    --disable-ldconfig               don't run ldconfig after installation (Linux only)
    --disable-verify                 don't obsolete
    --verbose                        run with verbose output


sudo ./install.sh --list-components

# Available components

* rustc
* rust-std-x86_64-unknown-linux-gnu
* rust-docs
* rust-docs-json-preview
* rust-demangler-preview
* cargo
* rustfmt-preview
* rls-preview
* rust-analyzer-preview
* llvm-tools-preview
* clippy-preview
* rust-analysis-x86_64-unknown-linux-gnu

"""
components_to_install = ["rustc", "cargo", "rust-std-x86_64-unknown-linux-gnu"]
app_dir = "/opt/rust"
app_sys_path = makepath(app_dir, "bin")

     
def install():
    zip_name = glob.glob("rust-*.tar.gz")[0]
    sub_dir = zip_name.rsplit(".tar", 1)[0]

    print(f"Installing: {zip_name} to: {app_dir}")
    
    # Get components list (corresponding to subfolders)
    with tarfile.open(zip_name, "r") as tar:
        file = tar.extractfile(f"{sub_dir}/components")
        if file:
            content = file.read()
    components = content.decode().splitlines()

    exclude_list = [f'{sub_dir}/{c}' for c in components if c and c not in components_to_install]
    extract_tar(zip_name, exclude_list=exclude_list)
    
    print(run(f'{makepath(sub_dir, "install.sh")} --prefix="{app_dir}" --components="{",".join([a for a in components_to_install])}"'))

    print('Linking rustc binary')
    if os.path.isfile('/opt/rust/bin/rustc'):
        print('ln -s /opt/rust/bin/rustc /usr/bin/rustc')
        remove_file('/usr/bin/rustc')
        run('ln -s /opt/rust/bin/rustc /usr/bin/rustc')

    print('Linking cargo binary')
    if os.path.isfile('/opt/rust/bin/cargo'):
        print('ln -s /opt/rust/bin/cargo /usr/bin/cargo')
        remove_file('/usr/bin/cargo')
        run('ln -s /opt/rust/bin/cargo /usr/bin/cargo')

def uninstall():
    if isfile(makepath(app_dir, "/lib/rustlib/uninstall.sh")):
        run(makepath(app_dir, "/lib/rustlib/uninstall.sh"))
    if isdir(app_dir):
        remove_tree(app_dir)
    remove_file('/usr/bin/cargo')
    remove_file('/usr/bin/rustc')

def extract_tar(
    tar_path: str,
    extract_path: str = ".",
    exclude_list: Optional[List[str]] = None,
    include_list: Optional[List[str]] = None
) -> None:
    """
    Extracts a tar archive to the given directory, with optional inclusion or exclusion filtering.

    :param tar_path: Path to the .tar (or .tar.gz, .tar.bz2, etc.) archive.
    :param extract_path: Directory where files should be extracted.
    :param exclude_list: List of path prefixes to exclude from extraction.
    :param include_list: List of path prefixes to include exclusively. Overrides exclude_list if provided.
    """
    with tarfile.open(tar_path, 'r:*') as tar:
        all_members = tar.getmembers()

        if include_list:
            include_set = set(include_list)
            members = [
                m for m in all_members
                if any(m.name.startswith(inc) for inc in include_set)
            ]
        elif exclude_list:
            exclude_set = set(exclude_list)
            members = [
                m for m in all_members
                if not any(m.name.startswith(exc) for exc in exclude_set)
            ]
        else:
            members = all_members

        tar.extractall(path=extract_path, members=members)
   