import { CommentFactory } from '../../factories/CommentFactory.mjs';
import { Metadata } from '../../schemas/metadata/Metadata.mjs';
import { PatternUtil } from '../../utils/PatternUtil.mjs';
import { json_schema_description } from './json_schema_description.mjs';
import { json_schema_plugin } from './json_schema_plugin.mjs';
import { json_schema_station } from './json_schema_station.mjs';
import { json_schema_title } from './json_schema_title.mjs';
import { metadata_to_pattern } from './metadata_to_pattern.mjs';

/** @internal */
const json_schema_object = (props) => json_schema_plugin({
    schema: emplace_object(props),
    tags: props.object.tags,
});
const emplace_object = (props) => {
    if (props.object.type.isLiteral() === true)
        return create_object_schema(props);
    const key = props.object.type.name;
    const $ref = `#/components/schemas/${key}`;
    if (props.components.schemas?.[key] !== undefined)
        return { $ref };
    const lazy = {};
    props.components.schemas ??= {};
    props.components.schemas[key] = lazy;
    Object.assign(lazy, create_object_schema(props));
    return { $ref };
};
/** @internal */
const create_object_schema = (props) => {
    // ITERATE PROPERTIES
    const properties = {};
    const extraMeta = {
        patternProperties: {},
        additionalProperties: undefined,
    };
    const required = [];
    for (const property of props.object.type.properties) {
        if (
        // FUNCTIONAL TYPE
        property.value.functions.length &&
            property.value.nullable === false &&
            property.value.isRequired() === true &&
            property.value.size() === 0)
            continue;
        else if (property.jsDocTags.find((tag) => tag.name === "hidden"))
            continue; // THE HIDDEN TAG
        const key = property.key.getSoleLiteral();
        const schema = json_schema_station({
            blockNever: true,
            components: props.components,
            attribute: {
                deprecated: property.jsDocTags.some((tag) => tag.name === "deprecated") ||
                    undefined,
                title: json_schema_title(property),
                description: json_schema_description(property),
            },
            metadata: property.value,
        });
        if (schema === null)
            continue;
        if (key !== null) {
            properties[key] = schema;
            if (property.value.isRequired() === true)
                required.push(key);
        }
        else {
            const pattern = metadata_to_pattern({
                top: true,
                metadata: property.key,
            });
            if (pattern === PatternUtil.STRING)
                extraMeta.additionalProperties = [property.value, schema];
            else
                extraMeta.patternProperties[pattern] = [property.value, schema];
        }
    }
    return {
        type: "object",
        properties,
        required,
        title: (() => {
            const info = props.object.type.jsDocTags.find((tag) => tag.name === "title");
            return info?.text?.length ? CommentFactory.merge(info.text) : undefined;
        })(),
        description: json_schema_description(props.object.type),
        additionalProperties: join({
            components: props.components,
            extra: extraMeta,
        }),
    };
};
/** @internal */
const join = (props) => {
    // LIST UP METADATA
    const elements = Object.values(props.extra.patternProperties || {});
    if (props.extra.additionalProperties)
        elements.push(props.extra.additionalProperties);
    // SHORT RETURN
    if (elements.length === 0)
        return undefined;
    else if (elements.length === 1)
        return elements[0][1];
    // MERGE METADATA AND GENERATE VULNERABLE SCHEMA
    const metadata = elements
        .map((tuple) => tuple[0])
        .reduce((x, y) => Metadata.merge(x, y));
    return (json_schema_station({
        blockNever: true,
        components: props.components,
        attribute: {},
        metadata,
    }) ?? undefined);
};

export { json_schema_object };
//# sourceMappingURL=json_schema_object.mjs.map
