"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LlmTypeCheckerV3_1 = void 0;
const OpenApiTypeCheckerBase_1 = require("./internal/OpenApiTypeCheckerBase");
/**
 * Type checker for LLM type schema v3.1.
 *
 * `LlmTypeCheckerV3_1` is a type checker of {@link ILlmSchemaV3_1}.
 *
 * @author Jeongho Nam - https://github.com/samchon
 */
var LlmTypeCheckerV3_1;
(function (LlmTypeCheckerV3_1) {
    /* -----------------------------------------------------------
      TYPE CHECKERS
    ----------------------------------------------------------- */
    /**
     * Test whether the schema is a nul type.
     *
     * @param schema Target schema
     * @returns Whether null type or not
     */
    LlmTypeCheckerV3_1.isNull = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isNull(schema);
    /**
     * Test whether the schema is an unknown type.
     *
     * @param schema Target schema
     * @returns Whether unknown type or not
     */
    LlmTypeCheckerV3_1.isUnknown = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isUnknown(schema);
    /**
     * Test whether the schema is a constant type.
     *
     * @param schema Target schema
     * @returns Whether constant type or not
     */
    LlmTypeCheckerV3_1.isConstant = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isConstant(schema);
    /**
     * Test whether the schema is a boolean type.
     *
     * @param schema Target schema
     * @returns Whether boolean type or not
     */
    LlmTypeCheckerV3_1.isBoolean = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isBoolean(schema);
    /**
     * Test whether the schema is an integer type.
     *
     * @param schema Target schema
     * @returns Whether integer type or not
     */
    LlmTypeCheckerV3_1.isInteger = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isInteger(schema);
    /**
     * Test whether the schema is a number type.
     *
     * @param schema Target schema
     * @returns Whether number type or not
     */
    LlmTypeCheckerV3_1.isNumber = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isNumber(schema);
    /**
     * Test whether the schema is a string type.
     *
     * @param schema Target schema
     * @returns Whether string type or not
     */
    LlmTypeCheckerV3_1.isString = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isString(schema);
    /**
     * Test whether the schema is an array type.
     *
     * @param schema Target schema
     * @returns Whether array type or not
     */
    LlmTypeCheckerV3_1.isArray = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isArray(schema);
    /**
     * Test whether the schema is an object type.
     *
     * @param schema Target schema
     * @returns Whether object type or not
     */
    LlmTypeCheckerV3_1.isObject = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isObject(schema);
    /**
     * Test whether the schema is a reference type.
     *
     * @param schema Target schema
     * @returns Whether reference type or not
     */
    LlmTypeCheckerV3_1.isReference = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isReference(schema);
    /**
     * Test whether the schema is an union type.
     *
     * @param schema Target schema
     * @returns Whether union type or not
     */
    LlmTypeCheckerV3_1.isOneOf = (schema) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isOneOf(schema);
    /**
     * Test whether the schema is recursive reference type.
     *
     * Test whether the target schema is a reference type, and test one thing more
     * that the reference is self-recursive or not.
     *
     * @param props Properties for recursive reference test
     * @returns Whether the schema is recursive reference type or not
     */
    LlmTypeCheckerV3_1.isRecursiveReference = (props) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.isRecursiveReference({
        prefix: "#/$defs/",
        components: {
            schemas: props.$defs,
        },
        schema: props.schema,
    });
    /* -----------------------------------------------------------
      OPERATORS
    ----------------------------------------------------------- */
    /**
     * Test whether the `x` schema covers the `y` schema.
     *
     * @param props Properties for testing
     * @returns Whether the `x` schema covers the `y` schema
     */
    LlmTypeCheckerV3_1.covers = (props) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.covers({
        prefix: "#/$defs/",
        components: {
            schemas: props.$defs,
        },
        x: props.x,
        y: props.y,
    });
    /**
     * Visit every nested schemas.
     *
     * Visit every nested schemas of the target, and apply the `props.closure`
     * function.
     *
     * Here is the list of occurring nested visitings:
     *
     * - {@link ILlmSchemaV3_1.IOneOf.oneOf}
     * - {@link ILlmSchemaV3_1.IReference}
     * - {@link ILlmSchemaV3_1.IObject.properties}
     * - {@link ILlmSchemaV3_1.IObject.additionalProperties}
     * - {@link ILlmSchemaV3_1.IArray.items}
     *
     * @param props Properties for visiting
     */
    LlmTypeCheckerV3_1.visit = (props) => OpenApiTypeCheckerBase_1.OpenApiTypeCheckerBase.visit({
        prefix: "#/$defs/",
        components: {
            schemas: props.$defs,
        },
        closure: props.closure,
        schema: props.schema,
    });
})(LlmTypeCheckerV3_1 || (exports.LlmTypeCheckerV3_1 = LlmTypeCheckerV3_1 = {}));
