import { ILlmSchemaV3 } from "../structures/ILlmSchemaV3";
/**
 * Type checker for LLM type schema v3.
 *
 * `LlmTypeCheckerV3` is a type checker of {@link ILlmSchemaV3}.
 *
 * @author Jeongho Nam - https://github.com/samchon
 */
export declare namespace LlmTypeCheckerV3 {
    /**
     * Visit every nested schemas.
     *
     * Visit every nested schemas of the target, and apply the `props.closure`
     * function.
     *
     * Here is the list of occurring nested visitings:
     *
     * - {@link ILlmSchemaV3.IOneOf.oneOf}
     * - {@link ILlmSchemaV3.IObject.additionalProperties}
     * - {@link ILlmSchemaV3.IArray.items}
     *
     * @param props Properties for visiting
     */
    const visit: (props: {
        closure: (schema: ILlmSchemaV3, accessor: string) => void;
        schema: ILlmSchemaV3;
        accessor?: string;
    }) => void;
    const covers: (x: ILlmSchemaV3, y: ILlmSchemaV3) => boolean;
    /**
     * Test whether the schema is an union type.
     *
     * @param schema Target schema
     * @returns Whether union type or not
     */
    const isOneOf: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.IOneOf;
    /**
     * Test whether the schema is an object type.
     *
     * @param schema Target schema
     * @returns Whether object type or not
     */
    const isObject: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.IObject;
    /**
     * Test whether the schema is an array type.
     *
     * @param schema Target schema
     * @returns Whether array type or not
     */
    const isArray: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.IArray;
    /**
     * Test whether the schema is a boolean type.
     *
     * @param schema Target schema
     * @returns Whether boolean type or not
     */
    const isBoolean: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.IBoolean;
    /**
     * Test whether the schema is an integer type.
     *
     * @param schema Target schema
     * @returns Whether integer type or not
     */
    const isInteger: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.IInteger;
    /**
     * Test whether the schema is a number type.
     *
     * @param schema Target schema
     * @returns Whether number type or not
     */
    const isNumber: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.INumber;
    /**
     * Test whether the schema is a string type.
     *
     * @param schema Target schema
     * @returns Whether string type or not
     */
    const isString: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.IString;
    /**
     * Test whether the schema is a null type.
     *
     * @param schema Target schema
     * @returns Whether null type or not
     */
    const isNullOnly: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.INullOnly;
    /**
     * Test whether the schema is a nullable type.
     *
     * @param schema Target schema
     * @returns Whether nullable type or not
     */
    const isNullable: (schema: ILlmSchemaV3) => boolean;
    /**
     * Test whether the schema is an unknown type.
     *
     * @param schema Target schema
     * @returns Whether unknown type or not
     */
    const isUnknown: (schema: ILlmSchemaV3) => schema is ILlmSchemaV3.IUnknown;
}
