"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LlmDataMerger = void 0;
/**
 * Data combiner for LLM function call.
 *
 * @author Samchon
 */
var LlmDataMerger;
(function (LlmDataMerger) {
    /**
     * Combine LLM and human arguments into one.
     *
     * When you composes {@link IOpenAiDocument} with
     * {@link IOpenAiDocument.IOptions.separate} option, then the arguments of the
     * target function would be separated into two parts; LLM (Large Language
     * Model) and human.
     *
     * In that case, you can combine both LLM and human composed arguments into
     * one by utilizing this {@link LlmDataMerger.parameters} function, referencing
     * the target function metadata {@link IOpenAiFunction.separated}.
     *
     * @param props Properties to combine LLM and human arguments with metadata.
     * @returns Combined arguments
     */
    LlmDataMerger.parameters = (props) => {
        const separated = props.function.separated;
        if (separated === undefined)
            throw new Error("Error on OpenAiDataComposer.parameters(): the function parameters are not separated.");
        return LlmDataMerger.value(props.llm, props.human);
    };
    /**
     * Combine two values into one.
     *
     * If both values are objects, then combines them in the properties level.
     *
     * Otherwise, returns the latter value if it's not null, otherwise the former
     * value
     *
     * - `return (y ?? x)`
     *
     * @param x Value X
     * @param y Value Y
     * @returns Combined value
     */
    LlmDataMerger.value = (x, y) => typeof x === "object" && typeof y === "object" && x !== null && y !== null
        ? combineObject(x, y)
        : Array.isArray(x) && Array.isArray(y)
            ? new Array(Math.max(x.length, y.length))
                .fill(0)
                .map((_, i) => LlmDataMerger.value(x[i], y[i]))
            : (y !== null && y !== void 0 ? y : x);
    const combineObject = (x, y) => {
        const output = Object.assign({}, x);
        for (const [k, v] of Object.entries(y))
            output[k] = LlmDataMerger.value(x[k], v);
        return output;
    };
})(LlmDataMerger || (exports.LlmDataMerger = LlmDataMerger = {}));
