import { IGeminiSchema } from "../structures/IGeminiSchema";
/**
 * Type checker for Gemini type schema.
 *
 * `GeminiTypeChecker` is a type checker of {@link IGeminiSchema}.
 *
 * @author Jeongho Nam - https://github.com/samchon
 */
export declare namespace GeminiTypeChecker {
    /**
     * Visit every nested schemas.
     *
     * Visit every nested schemas of the target, and apply the `props.closure`
     * function.
     *
     * Here is the list of occurring nested visitings:
     *
     * - {@link IGeminiSchema.IObject.properties}
     * - {@link IGeminiSchema.IArray.items}
     *
     * @param props Properties for visiting
     */
    const visit: (props: {
        closure: (schema: IGeminiSchema, accessor: string) => void;
        schema: IGeminiSchema;
        accessor?: string;
    }) => void;
    /**
     * Test whether the `x` schema covers the `y` schema.
     *
     * @param props Properties for testing
     * @returns Whether the `x` schema covers the `y` schema
     */
    const covers: (x: IGeminiSchema, y: IGeminiSchema) => boolean;
    /**
     * Test whether the schema is a boolean type.
     *
     * @param schema Target schema
     * @returns Whether boolean type or not
     */
    const isBoolean: (schema: IGeminiSchema) => schema is IGeminiSchema.IBoolean;
    /**
     * Test whether the schema is an integer type.
     *
     * @param schema Target schema
     * @returns Whether integer type or not
     */
    const isInteger: (schema: IGeminiSchema) => schema is IGeminiSchema.IInteger;
    /**
     * Test whether the schema is a number type.
     *
     * @param schema Target schema
     * @returns Whether number type or not
     */
    const isNumber: (schema: IGeminiSchema) => schema is IGeminiSchema.INumber;
    /**
     * Test whether the schema is a string type.
     *
     * @param schema Target schema
     * @returns Whether string type or not
     */
    const isString: (schema: IGeminiSchema) => schema is IGeminiSchema.IString;
    /**
     * Test whether the schema is an array type.
     *
     * @param schema Target schema
     * @returns Whether array type or not
     */
    const isArray: (schema: IGeminiSchema) => schema is IGeminiSchema.IArray;
    /**
     * Test whether the schema is an object type.
     *
     * @param schema Target schema
     * @returns Whether object type or not
     */
    const isObject: (schema: IGeminiSchema) => schema is IGeminiSchema.IObject;
    /**
     * Test whether the schema is a null type.
     *
     * @param schema Target schema
     * @returns Whether null type or not
     */
    const isNullOnly: (schema: IGeminiSchema) => schema is IGeminiSchema.INullOnly;
    /**
     * Test whether the schema is a nullable type.
     *
     * @param schema Target schema
     * @returns Whether nullable type or not
     */
    const isNullable: (schema: IGeminiSchema) => boolean;
    /**
     * Test whether the schema is an unknown type.
     *
     * @param schema Target schema
     * @returns Whether unknown type or not
     */
    const isUnknown: (schema: IGeminiSchema) => schema is IGeminiSchema.IUnknown;
}
