"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.uvRun = void 0;
exports.loopWhile = loopWhile;
exports.runLoopOnce = runLoopOnce;
exports.deasync = deasync;
exports.awaitSync = awaitSync;
/// <reference types="./missing-types" />
const binding_node_1 = require("./build/Release/binding.node");
Object.defineProperty(exports, "uvRun", { enumerable: true, get: function () { return binding_node_1.uvRun; } });
/**
 * Run tasks in the event loop until the `pred` returns true.
 *
 * @param pred A function that returns boolean loop condition.
 */
function loopWhile(pred) {
    while (pred()) {
        process._tickCallback();
        if (pred())
            (0, binding_node_1.uvRun)();
    }
}
/**
 * Run micro tasks until the micro task queue has been exhausted,
 * then run a macro task (if any).
 */
function runLoopOnce() {
    process._tickCallback();
    (0, binding_node_1.uvRun)(); // Macro task should be run after microtasks.
}
/**
 * Determine whether the value is a Promise.
 *
 * @see https://promisesaplus.com/
 */
function isThenable(value) {
    return typeof value?.then === "function";
}
// Can't use enum as async-to-sync breaks the control flow analyzing.
const Pending = 0;
const Fulfilled = 1;
const Rejected = 2;
/**
 * Generic wrapper of async function with conventional API signature
 * `function (...args, (error, result) => {})`.
 *
 * Returns `result` and throws `error` as exception if not null.
 *
 * @param fn the original callback style function
 * @return the wrapped function
 */
function deasync(fn) {
    return function (...args) {
        let state = Pending;
        let resultOrError;
        args.push((err, res) => {
            if (err) {
                resultOrError = err;
                state = Rejected;
            }
            else {
                resultOrError = res;
                state = Fulfilled;
            }
        });
        fn.apply(this, args);
        loopWhile(() => state === Pending);
        if (state === Rejected) {
            throw resultOrError;
        }
        else {
            return resultOrError;
        }
    };
}
/**
 * Similar with the keyword `await` but synchronously.
 *
 * @param promise A Promise or any value to wait for
 * @return Returns the fulfilled value of the promise, or the value itself if it's not a Promise.
 */
function awaitSync(promise) {
    let state = Pending;
    let resultOrError;
    if (!isThenable(promise)) {
        return promise;
    }
    promise.then(res => {
        resultOrError = res;
        state = Fulfilled;
    }, err => {
        resultOrError = err;
        state = Rejected;
    });
    loopWhile(() => state === Pending);
    if (state === Rejected) {
        throw resultOrError;
    }
    else {
        return resultOrError;
    }
}
//# sourceMappingURL=index.js.map