# -*- coding: utf-8 -*-
from setuphelpers import *
import re

bin_name = "wapttftpserver.bin"
service_name = "wapttftpserver"


def install():
    tftp_bin = os.path.join(WAPT.wapt_base_dir, bin_name)
    if service_is_enabled("wapttftpserver"):
        run("systemctl stop wapttftpserver")
    killalltasks(bin_name)
    filecopyto(bin_name, tftp_bin)
    run("chmod +x %s" % tftp_bin)

    # print("Apply wapttftpserver config")
    repo_path = inifile_readstring(WAPT.config_filename, "repo-sync", "local_repo_path")
    if not repo_path:
        if is_debian_based():
            repo_path = "/var/www"
        elif is_redhat_based():
            repo_path = "/var/www/html"
        else:
            makepath(WAPT.wapt_base_dir, "repository")
    tftp_path = makepath(repo_path, "wads", "pxe")
    mkdirs(tftp_path)

    if is_redhat_based():
        print("Add wapttftpserver firewall rule")
        setup_firewall()

        package_prefix = control.package.split("-", 1)[0]
        pkg_name = package_prefix + "-" + "remote-repo-nginx"
        if not WAPT.is_installed(pkg_name):
            # SELinux rules for CentOS/RedHat
            if re.match("^SELinux status:.*enabled", run("sestatus")):
                print("Redhat/Centos detected, tweaking SELinux rules")
                selinux_rules(repo_path)
                print("Nginx - SELinux correctly configured for Nginx reverse proxy")

    print("Create wapttftpserver service")

    wapttftpserver = rf'''
    [Unit]
Description=WAPT/WADS TFTP Server startup script
After=network-online.target

[Service]
Type=forking
RemainAfterExit=no
UMask=0027
#PIDFile=/run/wapttftpserver.pid
ExecStart=/opt/wapt/wapttftpserver.bin --fork --username=root --root_dir={tftp_path} --verbose --log-path=/dev/stdout
ExecStop=/opt/wapt/wapttftpserver.bin -k
TimeoutSec=300
SyslogIdentifier=wapttftpserver

[Install]
WantedBy=multi-user.target
'''

    with open('/usr/lib/systemd/system/wapttftpserver.service','w') as t:
        t.write(wapttftpserver)
        t.close

    if is_redhat_based():
        run("chcon -vR system_u:object_r:systemd_unit_file_t:s0  /usr/lib/systemd/system/wapttftpserver.service")

    run("chmod 644  /usr/lib/systemd/system/wapttftpserver.service")
    run("systemctl enable /usr/lib/systemd/system/wapttftpserver.service")
    run("systemctl restart wapttftpserver.service")


def uninstall():
    tftp_bin = os.path.join(WAPT.wapt_base_dir, bin_name)
    server_conf_dir = makepath(WAPT.wapt_base_dir, "conf")

    print("Disable wapttftpserver service")
    if service_is_enabled('wapttftpserver'):
        run("systemctl disable wapttftpserver")
    run("systemctl stop wapttftpserver")

    if isfile(tftp_bin):
        remove_file(tftp_bin)
    if len(glob.glob(server_conf_dir + os.sep + "*")) == 1:
        remove_tree(server_conf_dir)


def setup_firewall():
    """Add permanent rules for firewalld"""
    if not type_redhat():
        # only firewalld is supported right now
        return
    # check if firewalld is running
    if run("firewall-cmd --state", accept_returncodes=[0, 252]).strip() == "running":
        run("firewall-cmd --permanent --add-service=tftp")
        run("firewall-cmd --reload")
    else:
        run("firewall-offline-cmd --add-service=tftp")


def selinux_rules(local_repo):
    """SELinux httpd security rules"""
    run("setsebool -P httpd_can_network_connect 1")
    run("setsebool -P httpd_setrlimit on")
    for sepath in "wads":
        path = makepath(local_repo, sepath)
        mkdirs(path)
        try:
            run('semanage fcontext -a -t httpd_sys_content_t "%s(/.*)?"' % (path))
        except:
            try:
                run('semanage fcontext -m -t httpd_sys_content_t "%s(/.*)?"' % (path))
            except:
                pass
        run("restorecon -R -v %s" % (path))


def service_is_enabled(service):
    try:
        res = run("systemctl is-enabled wapttftpserver")
        return res.splitlines()[0] == "enabled"
    except:
        return False
