# -*- coding: utf-8 -*-
from setuphelpers import *
import platform


appx_package_name = "Microsoft.WindowsStore"
appx_dir = makepath(programfiles, "WindowsAppsInstallers")


def install():
    # Declare local variables
    bins_dir = control.package.split("-", 1)[1]
    old_appx_bins_dir = makepath(appx_dir, bins_dir)
    appx_bins_dir = makepath(os.getcwd(), bins_dir)

    # Removing sources because they are no longer needed
    if isdir(old_appx_bins_dir):
        print("Removing: %s" % (old_appx_bins_dir))
        remove_tree(old_appx_bins_dir)
    if dir_is_empty(appx_dir):
        print("Removing: %s since it is empty" % (appx_dir))
        remove_tree(appx_dir)

    # Removing binaries of different architectures
    for f in glob.glob(appx_bins_dir + "/*"):
        fname = f.split(os.sep)[-1]
        if control.architecture == "all":
            if "x86" in glob.glob(makepath(appx_bins_dir, f"{appx_package_name}_*"))[0].split(os.sep)[-1]:
                if not "x86" in fname and not "neutral" in fname:
                    remove_file(f)
            else:
                if not get_host_architecture() in fname and not "neutral" in fname:
                    remove_file(f)
        else:
            if not get_host_architecture() in fname and not "neutral" in fname:
                remove_file(f)
    bin_path = glob.glob(makepath(appx_bins_dir, f"{appx_package_name}_*"))[0]
    dependencies_pathes = ",".join([f'"{a}"' for a in glob.glob(makepath(appx_bins_dir, "*")) if not appx_package_name in a])

    # Installing Appx if needed
    add_appx_cmd = f'Add-AppxProvisionedPackage -Online -PackagePath "{bin_path}" -SkipLicense'
    if dependencies_pathes:
        add_appx_cmd += f" -DependencyPackagePath {dependencies_pathes}"

    appxprovisionedpackage = run_powershell(f'Get-AppXProvisionedPackage -Online | Where-Object DisplayName -Like "{appx_package_name}"')
    if appxprovisionedpackage is None:
        appxprovisionedpackage = {"Version": "0"}

    if Version(appxprovisionedpackage["Version"], 4) < Version(control.get_software_version(), 4) or force:
        print(f"Installing: {bin_path.split(os.sep)[-1]}")
        run_powershell(add_appx_cmd)
    else:
        print(f'{appxprovisionedpackage["PackageName"]} is already installed and up-to-date.')


def uninstall():
    # Declare local variables
    bins_dir = control.package.split("-", 1)[1]
    appx_bins_dir = makepath(appx_dir, bins_dir)

    print(f"Removing AppX: {appx_package_name}")
    remove_appx(appx_package_name)


def audit():
    # Declaring local variables
    audit_result = "OK"
    audit_version = True
    appxprovisionedpackage = run_powershell(f'Get-AppXProvisionedPackage -Online | Where-Object DisplayName -Like "{appx_package_name}"')

    # Auditing software
    if appxprovisionedpackage is None:
        print(f"{appx_package_name} is not installed.")
        audit_result = "ERROR"
    elif audit_version:
        if Version(appxprovisionedpackage.get("Version", "0"), 4) < Version(control.get_software_version(), 4):
            print(
                f'{appxprovisionedpackage["PackageName"]} is installed in version: {appxprovisionedpackage["Version"]} instead of: {control.get_software_version()}.'
            )
            audit_result = "WARNING"
        else:
            print(f'{appxprovisionedpackage["PackageName"]} is installed and up-to-date.')
    else:
        print(f'{appxprovisionedpackage["PackageName"]} is installed.')

    return audit_result


def remove_appx(package, default_user=True):
    """Remove Windows AppX package from the computer environment, excluding NonRemovable packages.

    Args:
        package (str): AppX package name. You can use an asterisk (*) as a wildcard.
        default_user (bool): Remove AppX package from the Windows image to prevent installation for new users.

    .. versionchanged:: 2.5
        No longer try to remove NonRemovable AppX package

    .. versionchanger:: 2.6
        No longer use run_powershell for uninstall
    """

    if running_as_admin() or running_as_system():
        if default_user:
            ps_script = f'Get-AppXProvisionedPackage -Online | Where-Object -Property DisplayName -Like {package} | Remove-AppxProvisionedPackage -Online -AllUsers'
            run_powershell_script(ps_script, output_format="text")

        ps_script = f'Get-AppxPackage -Name {package} -AllUsers | Where-Object -property NonRemovable -notlike True | Remove-AppxPackage -AllUsers'
        run_powershell_script(ps_script, output_format="text")

    else:
        ps_script = f'Get-AppxPackage -Name {package} | Where-Object -property NonRemovable -notlike True | Remove-AppxPackage'
        run_powershell_script(ps_script, output_format="text")


# The code below is temporary and may be removed once the WAPT PR has been merged.
def isARM64():
    return platform.machine().lower() in ('aarch64', 'arm64')

def get_host_architecture():
    if isARM64():
        return "arm64"
    elif isARM():
        return "arm"
    elif is64():
        return "x64"
    elif is32():
        return "x86"