# -*- coding: utf-8 -*-
from setuphelpers import *
import json
import requests
import time
import random
import string
import jinja2
import distro
from waptcrypto import SSLCertificate, SSLPrivateKey

# General configuration
## Directories
install_path = makepath('/opt')
mattermost_path = makepath(install_path,'mattermost')
mattermost_bin = makepath(mattermost_path,'bin','mattermost')
mmctl_path = makepath(mattermost_path,'.config','mmctl','config')
temp_dir = makepath('/tmp','mattermost-upgrade')

## Mattermost configuration
mattermost_url = 'https://url_of_mattermost_server'
instance_name = 'name'
use_nginx = True # Values : True / False
generate_ssl_cert = True # Values : True / False. Use False if you use commercial, Let's Encrypt or your PKI certificate. In this case, please configure "ssl_certificate_path" and "ssl_key_path" around line 220.


# WARNING #
# Before use this package, please configure mmctl on the mattermost server if it's not already set.
# See https://docs.mattermost.com/manage/mmctl-command-line-tool.html#mmctl-auth-set for more information


def check_version():

    if isfile(mmctl_path):
        print('Checking Mattermost Server installed version.')
        run('cd {} && ./bin/mmctl --config {} auth set {}'.format(mattermost_path,mmctl_path,instance_name))
        mattermost_raw_version = run('cd {} && ./bin/mmctl --config {} system version --json'.format(mattermost_path,mmctl_path))
    ##    {
    ##        "version": "6.3.1.6.3.1.dfe182cb916aace9a82e48375ccd364e.false"
    ##    }
        return str('.'.join(json.loads(mattermost_raw_version)['version'].split('.')[0:3]))

    else:
        print("It's appear you just install Mattermost Server, return control version")
        return str(control.version.split('-')[0])


def systemd_stop_service(servicename):
    print('Systemd stop %s' % servicename)
    run('systemctl stop %s' % servicename)


def systemd_start_service(servicename):
    print('Systemd start %s' % servicename)
    run('systemctl start %s' % servicename)


def install():
    print('Installing %s' % control.asrequirement())

    is_install = False
    is_upgrade = False

    if isdir(mattermost_path):
        if isfile(mattermost_bin):
            is_upgrade = True
            print('Mattermost server binary found !')
        else:
            is_upgrade = True
            print('Mattermost Server last install seem to be corrupted. The folder install found but not the binary. Continue anyway.')
    else:
        print('First install of Mattermost Server.')
        is_install = True

    # Upgrade
    if is_upgrade:
        mattermost_version = check_version()
        print('Mattermost server installed version is : %s' % mattermost_version)

        if Version(mattermost_version) < Version('10.0') and Version(mattermost_version) > Version('8.99'):
            if Version(mattermost_version) != Version(control.version.split('-')[0]):
                print('Staring upgrade.')

                if isdir(temp_dir):
                    remove_tree(temp_dir)

                print('Extracting Mattermost server archive in %s.' % temp_dir)
                run(r"tar -xf mattermost*.gz --transform='s,^[^/]\+,\0-upgrade,' -C /tmp")

                systemd_stop_service('mattermost')

                print('Backup previous installation.')
                run(r"cd {} && cp -ra mattermost/ mattermost-back-$(date +'%F-%H-%M')/".format(install_path))

                print('Remove old files.')
                run(r"cd {} && find mattermost/ mattermost/client/ -mindepth 1 -maxdepth 1 \! \( -type d \( -path mattermost/client -o -path mattermost/client/plugins -o -path mattermost/config -o -path mattermost/logs -o -path mattermost/plugins -o -path mattermost/data -o -path mattermost/.config \) -prune \) | sort | sudo xargs rm -r".format(install_path))

                print('Copy new files.')
                run(r"cd {} && cp -an /tmp/mattermost-upgrade/. mattermost/".format(install_path))

                print('Chown files')
                run('chown -R mattermost:mattermost {}'.format(mattermost_path))

                systemd_start_service('mattermost')

                print('Pause to wait 10 seconds to Mattermost server service start.')
                time.sleep(10)

                retry = 0
                return_code = 502
                while return_code != 200 or retry < 10:
                    retry += 1
                    print('Retry to connect on Mattermost Server : %i' % retry)
                    time.sleep(5)
                    r = requests.get('https://mattermost.tbnk.ca')
                    return_code = r.status_code

                if retry < 10:
                    if Version(check_version()) == Version(control.version.split('-')[0]):
                        print('Mattermost server seem to be correctly installed. Deleting temp upgrade dir.')
                        remove_tree(temp_dir)

            else:
                print('Mattermost is already in the same version like package version. Skipping.')
        else:
            print('Major release of Mattermost, please check official documentation before upgrade.')

    # Installation
    if is_install:
        print('Run install')

        print('Install requirements')
        install_apt('sudo')
        install_apt('curl')
        install_apt('gnupg')

        print('Install postgresql')
        database_password = ''.join(random.choices(string.ascii_uppercase + string.digits, k = 15))
        install_apt('postgresql postgresql-contrib')
        run('sudo -u postgres psql -c "CREATE DATABASE mattermost;"')
        run(""" sudo -u postgres psql -c "CREATE USER mmuser WITH PASSWORD '{}';" """.format(database_password))
        run('sudo -u postgres psql -c "GRANT ALL PRIVILEGES ON DATABASE mattermost to mmuser;"')
        run('sudo -u postgres psql -c "ALTER DATABASE mattermost OWNER TO mmuser;"')
        run('sudo -u postgres psql -c "GRANT USAGE, CREATE ON SCHEMA PUBLIC TO mmuser;"')

        print('Install Mattermost Server')

        print('Extracting Mattermost server archive in /tmp.')
        run(r"tar -xf mattermost*.gz -C /tmp")

        print('Copy Mattermost Server in %s' % mattermost_path)
        copytree2(makepath('/tmp','mattermost'),mattermost_path)
        mkdirs(makepath(mattermost_path,'data'))

        print('Add user mattermost')
        run(r'useradd --system --user-group mattermost')

        print('Chown files')
        run(r'chown -R mattermost:mattermost {}'.format(mattermost_path))
        run(r'chmod -R g+w {}'.format(mattermost_path))

        print('Configure config.json')
        with open(makepath(mattermost_path,'config','config.json'),'r+') as f:
            data = json.load(f)
            data['SqlSettings']['DriverName'] = 'postgres'
            data['SqlSettings']['DataSource'] = "postgres://mmuser:{}@127.0.0.1:5432/mattermost?sslmode=disable&connect_timeout=10".format(database_password)

            f.seek(0)
            json.dump(data, f)
            f.truncate()

        print('Create Mattermost server service')

        jinja_env = jinja2.Environment(loader=jinja2.FileSystemLoader('templates'))
        template = jinja_env.get_template('mattermost.service.j2')

        template_variables = {
            'database_backend': 'postgresql',
            'mattermost_path': mattermost_path
        }

        config_string = template.render(template_variables)
        print('Create Mattermost Server service configuration file %s' % makepath('/lib','systemd','system','mattermost.service'))
        with open(makepath('/lib','systemd','system','mattermost.service'), 'wt') as dst_file:
            dst_file.write(config_string)

        run(r'systemctl daemon-reload')

        systemd_start_service('mattermost')

        run('systemctl enable mattermost.service')

        print('Mattermost Server is installed !')

    # Install Nginx if needed
    if use_nginx:
        #WAPT.install('{}-nginx'.format(control.package.split('-')[0]))
        print('Install Nginx Web Server')
        install_apt('nginx')

        jinja_env = jinja2.Environment(loader=jinja2.FileSystemLoader('templates'))
        template = jinja_env.get_template('nginx.conf.j2')

        if distro.codename().lower() != 'buster':
            print('ssl_early_data supported')
            ssl_early_data_support = True
        else:
            print('ssl_early_data NOT supported')
            ssl_early_data_support = False

        if generate_ssl_cert:
            print('Generate SSL key and certificate')
            ap_ssl_dir = makepath(mattermost_path,'ssl')
            mkdirs(ap_ssl_dir)
            key_fn = os.path.join(ap_ssl_dir,'key.pem')
            key = SSLPrivateKey(key_fn)
            if not os.path.isfile(key_fn):
                print('Create SSL RSA Key %s' % key_fn)
                key.create()
                key.save_as_pem()

            cert_fn = os.path.join(ap_ssl_dir,'cert.pem')
            if os.path.isfile(cert_fn):
                crt = SSLCertificate(cert_fn)
                if os.path.isfile(cert_fn):
                    crt = SSLCertificate(cert_fn)
                    if crt.cn != get_fqdn():
                        os.rename(cert_fn,"%s-%s.old" % (cert_fn,'{:%Y%m%d-%Hh%Mm%Ss}'.format(datetime.datetime.now())))
                        crt = key.build_sign_certificate(cn=get_fqdn(),dnsname=get_fqdn(),is_code_signing=False)
                        print('Create X509 cert %s' % cert_fn)
                        crt.save_as_pem(cert_fn)
            else:
                crt = key.build_sign_certificate(cn=get_fqdn(),dnsname=get_fqdn(),is_code_signing=False)
                print('Create X509 cert %s' % cert_fn)
                crt.save_as_pem(cert_fn)

            template_variables = {
                'mattermost_url': mattermost_url.replace('https://','').replace('http://',''),
                'ssl_certificate_path': cert_fn.replace('\\','/'),
                'ssl_key_path': key_fn.replace('\\','/'),
                'ssl_early_data_support': ssl_early_data_support
            }
        else:
            template_variables = {
                'mattermost_url': mattermost_url.replace('https://','').replace('http://',''),
                'ssl_certificate_path': makepath('/etc','ssl','cert.pem'),
                'ssl_key_path': makepath('/etc','ssl','key.pem'),
                'ssl_early_data_support': ssl_early_data_support
            }


        config_string = template.render(template_variables)
        print('Create Nginx configuration file %s' % makepath('/etc','nginx','sites-avalables','mattermost.conf'))
        with open(makepath('/etc','nginx','sites-available','mattermost.conf'), 'wt') as dst_file:
            dst_file.write(config_string)

        print('Create symlink to activate mattermost site')
        if not os.path.exists('/etc/nginx/sites-enabled/mattermost.conf'):
            print(run('ln -s /etc/nginx/sites-available/mattermost.conf /etc/nginx/sites-enabled/mattermost.conf'))

        print('Test Nginx configuration and start nginx')
        print(run('nginx -t'))
        print(run('systemctl restart nginx'))
        print(run('systemctl enable nginx'))


def uninstall():
    print('Uninstall Mattermost Server')
    print("Warning ! This script doesn't remove Mysql Server or Postgresql Server")
    systemd_stop_service('mattermost')

    print('Remove nginx configuration files')
    os.unlink(makepath('/etc','nginx','sites-enabled','mattermost.conf'))
    remove_file(makepath('/etc','nginx','sites-avalables','mattermost.conf'))
    print(run('nginx -t'))
    print(run('systemctl restart nginx'))

    print('Remove Mattermost Server Service')
    remove_file(makepath('/lib','systemd','system','mattermost.service'))
    run('systemctl daemon-reload')

    print('Remove Mattermost server folder')
    remove_tree(mattermost_path)


def audit():
    installed_version = check_version()

    if Version(installed_version) == Version(control.version.split('-')[0]):
        print('Mattermost server version : %s' % installed_version)
        return "OK"
    else:
        print('Mattermost server version is %s not corresponding to package version.' % installed_version)
        return "ERROR"
