# -*- coding: utf-8 -*-
from setuphelpers import *
import string
import random
import jinja2

from waptcrypto import SSLCertificate, SSLPrivateKey

kc_url = 'https://keycloak.mydomain.lan' # Values : URL String
kc_health = 'true' # Values (str) : true / false
kc_metrics = 'false' # Values (str) : true / false
use_apache2 = True # Values : True / False

generate_ssl_cert = True # Values : True / False. Use False if you use commercial, Let's Encrypt or your PKI certificate.
# In this case, please configure "ssl_certificate_path" and "ssl_key_path".
ssl_certificate_path = None # Set None if generate_ssl_cert = True
ssl_key_path = None # Set None if generate_ssl_cert = True

keycloak_installdir = makepath('/opt','keycloak')
keycloak_installdir_bck = makepath('/opt','keycloak.bck')
keytabfile_path = makepath(keycloak_installdir,'conf','http-krb5.keytab')
keytabfile_path_bck = makepath(keycloak_installdir_bck,'conf','http-krb5.keytab')
keycloak_a2site = makepath('/etc','apache2','sites-available','keycloak.conf')


def systemd_enable_start_service(servicename):
    return run('LANG=C systemctl enable --now %s' % (servicename))


def install():

    print('If upgrade, stop keycloak service')
    try:
        systemd_stop_service('keycloak.service')
        first_install = False
    except:
        print("It's the first install")
        first_install = True

    print('Install requirements')   
    

    if get_distrib_linux() == 'debian' and Version(get_distrib_version()) >= Version("13"):
        install_apt('openjdk-21-jdk')
    else:
        install_apt('openjdk-17-jdk')
        
    install_apt('postgresql')

    if not first_install:
        if isdir(keycloak_installdir_bck):
            print('Remove old backup dir')
            remove_tree(keycloak_installdir_bck)

        print('Create backup of old installation.')
        copytree2(keycloak_installdir,keycloak_installdir_bck)
        remove_tree(keycloak_installdir)

    mkdirs(keycloak_installdir)

    print('Extract archive')
    version = control.version.split('-')[0]
    run(f'tar xzf keycloak-{version}.tar.gz -C {keycloak_installdir}')
    # Patch to use keycloak folder without version number in directory name.
    copytree2(makepath(keycloak_installdir,f'keycloak-{version}'),keycloak_installdir)
    remove_tree(makepath(keycloak_installdir,f'keycloak-{version}'))

    if first_install:
        print('Create group and user')
        run('groupadd -r keycloak')
        run(f'useradd -r -g keycloak -d {keycloak_installdir} -s /sbin/nologin keycloak')

        print('Create Postgresql user and database')
        run('sudo -u postgres psql -c "create database keycloak with encoding \'UTF8\';" ')

        database_password = ''.join(random.choices(string.ascii_uppercase + string.digits, k = 15))
        run(f""" sudo -u postgres psql -c "create user keycloak with password '{database_password}';" """)

        run(f""" sudo -u postgres psql -c "grant all ON database keycloak to keycloak;" """)
        run(f""" sudo -u postgres psql -d keycloak -c "grant all on schema public to keycloak ;" """)

        print('Create service')
        service = f"""[Unit]
Description=Keycloak Service
After=network.target syslog.target postgresql.target

[Service]
User=keycloak
Group=keycloak
ExecStart={keycloak_installdir}/bin/kc.sh start
SuccessExitStatus=0 143
WorkingDirectory={keycloak_installdir}

[Install]
WantedBy=multi-user.target

"""
        with open('/lib/systemd/system/keycloak.service','w') as service_file:
            service_file.write(service)

        run('systemctl daemon-reload')

    print('Set user rights')
    run(f'chown -R keycloak:keycloak {keycloak_installdir}')

    if first_install:
        if use_apache2:
            print('Install Apache2')
            install_apt('apache2')

            print('Enable Apache2 modules')
            run('a2enmod ssl headers proxy proxy_http')

            jinja_env = jinja2.Environment(loader=jinja2.FileSystemLoader('templates'))
            template = jinja_env.get_template('apache2.conf.j2')

            if generate_ssl_cert:
                print('Generate SSL key and certificate')
                ap_ssl_dir = makepath('/etc','ssl')
                mkdirs(ap_ssl_dir)
                key_fn = os.path.join(ap_ssl_dir,'private','keycloak_key.pem')
                key = SSLPrivateKey(key_fn)
                if not os.path.isfile(key_fn):
                    print('Create SSL RSA Key %s' % key_fn)
                    key.create()
                    key.save_as_pem()

                cert_fn = os.path.join(ap_ssl_dir,'certs','keycloak_cert.pem')
                if os.path.isfile(cert_fn):
                    crt = SSLCertificate(cert_fn)
                    if os.path.isfile(cert_fn):
                        crt = SSLCertificate(cert_fn)
                        if crt.cn != get_fqdn():
                            os.rename(cert_fn,"%s-%s.old" % (cert_fn,'{:%Y%m%d-%Hh%Mm%Ss}'.format(datetime.datetime.now())))
                            crt = key.build_sign_certificate(cn=get_fqdn(),is_code_signing=False)
                            print('Create X509 cert %s' % cert_fn)
                            crt.save_as_pem(cert_fn)
                else:
                    crt = key.build_sign_certificate(cn=get_fqdn(),is_code_signing=False)
                    print('Create X509 cert %s' % cert_fn)
                    crt.save_as_pem(cert_fn)

                template_variables = {
                    'kc_url': kc_url.replace('https://','').replace('http://',''),
                    'ssl_certificate_path': cert_fn.replace('\\','/'),
                    'ssl_key_path': key_fn.replace('\\','/')
                }
            else:
                template_variables = {
                    'kc_url': kc_url.replace('https://','').replace('http://',''),
                    'ssl_certificate_path': ssl_certificate_path,
                    'ssl_key_path': ssl_key_path
                }

            print('Write Apache2 site configfile')
            config_string = template.render(template_variables)
            print(f'Create Apache2 configuration file {keycloak_a2site}')
            with open(keycloak_a2site, 'w') as dst_file:
                dst_file.write(config_string)

            print('Enable keycloak site and disable default site')
            run('a2ensite keycloak.conf')
            run_notfatal('a2dissite 000-default.conf')

            print('Restart apache2 service')
            systemd_restart_service('apache2')


        jinja_env = jinja2.Environment(loader=jinja2.FileSystemLoader('templates'))
        template = jinja_env.get_template('keycloak.conf.j2')

        template_variables = {
            'use_apache2': use_apache2,
            'pgsql_passwd': database_password,
            'kc_health': kc_health,
            'kc_metrics': kc_metrics,
            'kc_url': kc_url.replace('https://','').replace('http://',''),
        }

        print('Write Keycloak configuration')
        config_string = template.render(template_variables)
        print('Create Keycloak configuration file %s' % makepath(keycloak_installdir,'conf/keycloak.conf'))
        with open(makepath(keycloak_installdir,'conf/keycloak.conf'), 'w') as dst_file:
                dst_file.write(config_string)

        print('Start Keycloak Server to create first admin user. Please wait 120 seconds.')
        run_notfatal('export KEYCLOAK_ADMIN=admin && export KEYCLOAK_ADMIN_PASSWORD=admin && /opt/keycloak/bin/kc.sh start', timeout=120)

    if not first_install:
        print('Copy old configfile')
        filecopyto(makepath(keycloak_installdir_bck,'conf/keycloak.conf'), makepath(keycloak_installdir,'conf','keycloak.conf'))
        if isfile(keytabfile_path_bck):
            filecopyto(keytabfile_path_bck,keytabfile_path)

    print('Build Keycloak for production')
    run("/opt/keycloak/bin/kc.sh build")

    print('Set user rights')
    run(f'chown -R keycloak:keycloak {keycloak_installdir}')
    if isdir('/var/log/keycloak'):
        run(f'chown -R keycloak:keycloak /var/log/keycloak')

    print('Start Keycloak service')
    systemd_enable_start_service('keycloak')


def uninstall():

    print('Stop keycloak service')
    systemd_stop_service('keycloak')

    print('Delete Postgresql user and database')
    run('sudo -u postgres psql -c "drop database keycloak;" ')
    run('sudo -u postgres psql -c "drop user if exists keycloak;" ')

    print('Remove keycloak directory')
    remove_tree(keycloak_installdir)

    print('Disable apache configuration')
    run('a2dissite keycloak.conf')

    print('Restart apache2 service')
    systemd_restart_service('apache2')

    print('Uninstall done, but not for postgresql and openjdk-25-jdk')