import { getClient, debug, uuid4 } from '@sentry/core';
import { ipcChannelUtils, RENDERER_ID_HEADER } from '../common/ipc.js';

/* eslint-disable no-restricted-globals */
/* eslint-disable no-console */
/** Gets the available IPC implementation */
function getImplementation(ipcKey) {
    const ipcUtil = ipcChannelUtils(ipcKey);
    // Favour IPC if it's been exposed by a preload script
    if (window.__SENTRY_IPC__?.[ipcUtil.namespace]) {
        return window.__SENTRY_IPC__[ipcUtil.namespace];
    }
    else {
        debug.log('IPC was not configured in preload script, falling back to custom protocol and fetch');
        // A unique ID used to identify this renderer and is send in the headers of every request
        // Because it added as a global, this can be fetched from the main process via executeJavaScript
        const id = (window.__SENTRY_RENDERER_ID__ = uuid4());
        const headers = { [RENDERER_ID_HEADER]: id };
        return {
            sendRendererStart: () => {
                fetch(ipcUtil.createUrl('start'), { method: 'POST', body: '', headers }).catch(() => {
                    console.error(`Sentry SDK failed to establish connection with the Electron main process.
  - Ensure you have initialized the SDK in the main process
  - If your renderers use custom sessions, be sure to set 'getSessions' in the main process options
  - If you are bundling your main process code and using Electron < v5, you'll need to manually configure a preload script`);
                });
            },
            sendScope: (body) => {
                fetch(ipcUtil.createUrl('scope'), { method: 'POST', body, headers }).catch(() => {
                    // ignore
                });
            },
            sendEnvelope: (body) => {
                fetch(ipcUtil.createUrl('envelope'), { method: 'POST', body, headers }).catch(() => {
                    // ignore
                });
            },
            sendStatus: (status) => {
                fetch(ipcUtil.createUrl('status'), {
                    method: 'POST',
                    body: JSON.stringify({ status }),
                    headers,
                }).catch(() => {
                    // ignore
                });
            },
            sendStructuredLog: (log) => {
                fetch(ipcUtil.createUrl('structured-log'), {
                    method: 'POST',
                    body: JSON.stringify(log),
                    headers,
                }).catch(() => {
                    // ignore
                });
            },
        };
    }
}
let cachedInterfaces;
/**
 * Renderer IPC interface
 *
 * Favours IPC if its been exposed via a preload script but will
 * fallback to custom protocol and fetch if IPC is not available
 */
function getIPC(client = getClient()) {
    if (!client) {
        throw new Error('Could not find client, make sure to call Sentry.init before getIPC');
    }
    if (!cachedInterfaces) {
        cachedInterfaces = new WeakMap();
    }
    const found = cachedInterfaces.get(client);
    if (found) {
        return found;
    }
    const namespace = client.getOptions().ipcNamespace;
    const implementation = getImplementation(namespace);
    cachedInterfaces.set(client, implementation);
    implementation.sendRendererStart();
    return implementation;
}

export { getIPC };//# sourceMappingURL=http://go/sourcemap/sourcemaps/3578107fdf149b00059ddad37048220e41681000/node_modules/@sentry/electron/esm/renderer/ipc.js.map
