const core = require('@sentry/core');
const electron = require('electron');

let RENDERERS;
/**
 * Enable tracking of renderer properties via events
 *
 * This allows us to get the last known state of a renderer even if it's been destroyed
 **/
function trackRendererProperties() {
    if (RENDERERS) {
        return;
    }
    const renderers = (RENDERERS = new Map());
    function updateUrl(id, url) {
        const state = renderers.get(id) || { id };
        state.url = core.normalizeUrlToBase(url, electron.app.getAppPath());
        renderers.set(id, state);
    }
    function updateTitle(id, title) {
        const state = renderers.get(id) || { id };
        state.title = title;
        renderers.set(id, state);
    }
    electron.app.on('web-contents-created', (_, contents) => {
        const id = contents.id;
        contents.on('did-navigate', (_, url) => updateUrl(id, url));
        contents.on('did-navigate-in-page', (_, url) => updateUrl(id, url));
        contents.on('page-title-updated', (_, title) => updateTitle(id, title));
        contents.on('destroyed', () => {
            // We need to delay since consumers of this API sometimes need to
            // access the state shortly after a renderer is destroyed
            setTimeout(() => {
                renderers.delete(id);
            }, 5000);
        });
    });
}
/** Gets the properties for a renderer */
function getRendererProperties(id) {
    return RENDERERS?.get(id);
}

exports.getRendererProperties = getRendererProperties;
exports.trackRendererProperties = trackRendererProperties;//# sourceMappingURL=http://go/sourcemap/sourcemaps/56f0a83df8e9eb48585fcc4858a9440db4cc7770/node_modules/@sentry/electron/main/renderers.js.map
