import { createStackParser, nodeStackLineParser, debug } from '@sentry/core';
import { createGetModuleFromFilename } from '@sentry/node';
import { app } from 'electron';
import { electronRendererStackParser } from '../renderer/stack-parse.js';
import { ELECTRON_MAJOR_VERSION } from './electron-normalize.js';

// node.js stack parser but filename normalized before parsing the module
const defaultStackParser = createStackParser(nodeStackLineParser(createGetModuleFromFilename(app.getAppPath())));
/**
 * Captures stack frames from a renderer process
 *
 * Requires Electron >= 34 and throws an error on older versions
 *
 * @param webContents The WebContents to capture stack frames from
 * @returns A promise that resolves to an array of Sentry StackFrames
 */
async function captureRendererStackFrames(webContents) {
    if (ELECTRON_MAJOR_VERSION < 34) {
        throw new Error('Electron >= 34 required to capture stack frames via `frame.collectJavaScriptCallStack()`');
    }
    if (webContents.isDestroyed()) {
        return undefined;
    }
    const frame = webContents.mainFrame;
    const stack = await frame.collectJavaScriptCallStack();
    if (!stack) {
        return undefined;
    }
    if (stack.includes('Website owner has not opted in')) {
        debug.warn("Could not collect renderer stack frames.\nA 'Document-Policy' header of 'include-js-call-stacks-in-crash-reports' must be set");
        return undefined;
    }
    return electronRendererStackParser(stack);
}

export { captureRendererStackFrames, defaultStackParser };//# sourceMappingURL=http://go/sourcemap/sourcemaps/ce371ffbf5e240ca47f4b5f3f20efed084991120/node_modules/@sentry/electron/esm/main/stack-parse.js.map
